<?php

namespace App\Console\Commands;

use App\Models\Financeiro2;
use App\Models\User;
use App\Services\JurosService;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

/**
 * Command para calcular juros automáticos de parcelas vencidas
 * Migrado do sistema legado: /legado/crons/status.php
 */
class CalcularJurosCommand extends Command
{
    protected $signature = 'juros:calcular {--dry-run : Simula o cálculo sem salvar}';
    protected $description = 'Calcula juros automáticos para parcelas vencidas baseado na configuração de cada usuário';

    private JurosService $jurosService;

    public function __construct(JurosService $jurosService)
    {
        parent::__construct();
        $this->jurosService = $jurosService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Iniciando cálculo de juros automáticos...');
        
        $dryRun = $this->option('dry-run');
        
        if ($dryRun) {
            $this->warn('Modo simulação ativado - nenhuma alteração será salva.');
        }

        try {
            $dataAtual = now();
            $totalProcessadas = 0;
            $totalComJuros = 0;
            $totalJurosAplicados = 0;

            // Busca parcelas pendentes (vencidas)
            $parcelas = Financeiro2::where(function($query) {
                $query->where('pagoem', 'n')
                      ->orWhere('status', '1');
            })->get();

            $this->info("Encontradas {$parcelas->count()} parcelas pendentes.");

            foreach ($parcelas as $parcela) {
                $totalProcessadas++;

                try {
                    // Verifica se a parcela está vencida
                    $dataVencimento = Carbon::createFromFormat('d/m/Y', $parcela->datapagamento);
                    
                    if ($dataVencimento->isFuture()) {
                        // Ainda não venceu
                        continue;
                    }

                    // Calcula dias de atraso
                    $diasAtraso = $dataAtual->diffInDays($dataVencimento);
                    
                    if ($diasAtraso <= 0) {
                        // Vence hoje ou não está em atraso
                        continue;
                    }

                    // Busca configuração do master
                    $master = User::find($parcela->idm);
                    if (!$master) {
                        continue;
                    }

                    // Verifica se tem juros configurados
                    $taxaJurosDiaria = $master->juros_diarios ?? 0;
                    
                    if ($taxaJurosDiaria <= 0) {
                        // Usuário não tem juros configurados
                        continue;
                    }

                    // Verifica se juros já foram calculados
                    if ($parcela->juros_calculados == 1) {
                        // Juros já aplicados, mas pode precisar atualizar
                        // Recalcula se os dias de atraso mudaram
                        if ($parcela->dias_vencidos == $diasAtraso) {
                            continue;
                        }
                    }

                    // Calcula juros
                    $valorOriginal = $this->getValorOriginal($parcela);
                    $valorComJuros = $this->jurosService->calcularJuros(
                        $valorOriginal,
                        $taxaJurosDiaria,
                        $diasAtraso
                    );

                    $jurosAplicado = $valorComJuros - $valorOriginal;

                    $this->line("Parcela #{$parcela->id}: Valor R$ {$valorOriginal} → R$ {$valorComJuros} (Juros: R$ {$jurosAplicado} - {$diasAtraso} dias)");

                    // Atualiza parcela
                    if (!$dryRun) {
                        $parcela->update([
                            'parcela' => $valorComJuros,
                            'juros_calculados' => 1,
                            'taxa_juros_diaria' => $taxaJurosDiaria,
                            'dias_vencidos' => $diasAtraso,
                        ]);
                    }

                    $totalComJuros++;
                    $totalJurosAplicados += $jurosAplicado;

                } catch (\Exception $e) {
                    Log::error('Erro ao calcular juros da parcela', [
                        'parcela_id' => $parcela->id,
                        'error' => $e->getMessage()
                    ]);
                    continue;
                }
            }

            $this->info("\n" . str_repeat('=', 60));
            $this->info("Cálculo de juros concluído:");
            $this->info("- Parcelas processadas: {$totalProcessadas}");
            $this->info("- Parcelas com juros aplicados: {$totalComJuros}");
            $this->info("- Total de juros: R$ " . number_format($totalJurosAplicados, 2, ',', '.'));
            $this->info(str_repeat('=', 60));

            if ($dryRun) {
                $this->warn("\nSimulação concluída - nenhuma alteração foi salva.");
            }
            
            return Command::SUCCESS;

        } catch (\Exception $e) {
            Log::error('Erro ao calcular juros', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            $this->error('Erro: ' . $e->getMessage());
            return Command::FAILURE;
        }
    }

    /**
     * Obtém o valor original da parcela (sem juros)
     */
    private function getValorOriginal(Financeiro2 $parcela): float
    {
        // Se juros já foram calculados, busca o valor original
        if ($parcela->juros_calculados == 1) {
            // Precisa buscar o valor original no financeiro1
            $financeiro1 = \App\Models\Financeiro1::where('chave', $parcela->chave)->first();
            if ($financeiro1) {
                return (float) $financeiro1->vparcela;
            }
        }

        // Retorna o valor atual da parcela
        return (float) $parcela->parcela;
    }

    /**
     * Recalcula juros de todas as parcelas (comando de manutenção)
     */
    public function recalcularTodos(): void
    {
        $this->info('Recalculando juros de todas as parcelas...');

        $parcelas = Financeiro2::where('juros_calculados', 1)->get();

        foreach ($parcelas as $parcela) {
            $parcela->update([
                'juros_calculados' => 0,
                'taxa_juros_diaria' => 0,
                'dias_vencidos' => 0,
            ]);
        }

        $this->info('Flags de juros zeradas. Execute o comando novamente para recalcular.');
    }
}
