<?php

namespace App\Console\Commands;

use App\Models\Financeiro2;
use App\Models\Financeiro1;
use App\Models\FinanceiroGlobal;
use App\Models\Cliente;
use App\Models\User;
use App\Models\Mensagem;
use App\Models\MercadoPago;
use App\Services\WhatsAppService;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

/**
 * Command para enviar cobranças automáticas via WhatsApp
 * Migrado do sistema legado: /legado/crons/cob_auto.php
 */
class CobrancaAutomaticaCommand extends Command
{
    protected $signature = 'cobranca:automatica';
    protected $description = 'Envia cobranças automáticas via WhatsApp baseado nas configurações de mensagens';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Iniciando envio de cobranças automáticas...');

        try {
            $horaAtual = now()->format('H:i');
            $dataAtual = now()->format('Ymd');
            
            // Calcula datas de notificação
            $datas = [
                'vencendo_7_dias' => now()->addDays(7)->format('Ymd'),
                'vencendo_5_dias' => now()->addDays(5)->format('Ymd'),
                'vencendo_3_dias' => now()->addDays(3)->format('Ymd'),
                'vence_hoje' => $dataAtual,
            ];

            // Busca parcelas pendentes
            $parcelas = Financeiro2::where('pagoem', 'n')
                ->orWhere('status', '1')
                ->get();

            $totalProcessadas = 0;
            $totalEnviadas = 0;

            foreach ($parcelas as $parcela) {
                $totalProcessadas++;
                
                // Converte data de pagamento para comparação
                $dataVencimento = Carbon::createFromFormat('d/m/Y', $parcela->datapagamento)->format('Ymd');
                
                // Verifica se a parcela está no intervalo de notificação
                if (!$this->deveNotificar($dataVencimento, $datas, $dataAtual)) {
                    continue;
                }

                // Busca dados do master/usuário
                $master = User::find($parcela->idm);
                if (!$master) {
                    continue;
                }

                // Verifica configuração do WhatsApp
                if (!$master->whatsapp_api || !$master->tokenapi) {
                    continue;
                }

                // Busca dados do cliente
                $cliente = Cliente::find($parcela->idc);
                if (!$cliente) {
                    continue;
                }

                // Determina tipo de mensagem baseado na data
                $tipoMensagem = $this->getTipoMensagem($dataVencimento, $datas, $dataAtual);
                
                // Busca mensagem configurada
                $mensagemConfig = Mensagem::where('tipo', $tipoMensagem)
                    ->where('status', '1')
                    ->where('idu', $parcela->idm)
                    ->first();

                if (!$mensagemConfig) {
                    continue;
                }

                // Verifica se é o horário configurado para envio
                if ($mensagemConfig->hora !== $horaAtual) {
                    continue;
                }

                // Envia cobrança
                if ($this->enviarCobranca($parcela, $cliente, $master, $mensagemConfig)) {
                    $totalEnviadas++;
                    $this->info("Cobrança enviada: Cliente {$cliente->nome} - Parcela R$ {$parcela->parcela}");
                }
            }

            $this->info("Processamento concluído: {$totalProcessadas} parcelas processadas, {$totalEnviadas} cobranças enviadas.");
            
            return Command::SUCCESS;

        } catch (\Exception $e) {
            Log::error('Erro ao executar cobrança automática', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            $this->error('Erro: ' . $e->getMessage());
            return Command::FAILURE;
        }
    }

    /**
     * Verifica se a parcela deve ser notificada
     */
    private function deveNotificar(string $dataVencimento, array $datas, string $dataAtual): bool
    {
        // Vencendo em 7, 5 ou 3 dias
        if (in_array($dataVencimento, array_values($datas))) {
            return true;
        }

        // Vence hoje
        if ($dataVencimento === $dataAtual) {
            return true;
        }

        // Já venceu (atraso)
        if ($dataVencimento < $dataAtual) {
            return true;
        }

        return false;
    }

    /**
     * Determina o tipo de mensagem baseado na data
     */
    private function getTipoMensagem(string $dataVencimento, array $datas, string $dataAtual): int
    {
        if ($dataVencimento === $datas['vencendo_7_dias']) {
            return 7; // 7 dias antes
        }
        
        if ($dataVencimento === $datas['vencendo_5_dias']) {
            return 1; // 5 dias antes
        }
        
        if ($dataVencimento === $datas['vencendo_3_dias']) {
            return 2; // 3 dias antes
        }
        
        if ($dataVencimento === $dataAtual) {
            return 3; // No dia
        }
        
        if ($dataVencimento < $dataAtual) {
            return 4; // Vencido
        }

        return 0;
    }

    /**
     * Envia cobrança via WhatsApp
     */
    private function enviarCobranca(
        Financeiro2 $parcela, 
        Cliente $cliente, 
        User $master, 
        Mensagem $mensagemConfig
    ): bool {
        try {
            // Inicializa serviço WhatsApp
            $whatsapp = new WhatsAppService(
                $master->whatsapp_api,
                $master->whatsapp_token ?? $master->tokenapi,
                $master->tokenapi
            );

            // Verifica se está conectado
            $status = $whatsapp->getConnectionStatus();
            if (!$status['connected']) {
                Log::warning('WhatsApp não conectado', ['user_id' => $master->id]);
                return false;
            }

            // Prepara dados para variáveis
            $link = $this->gerarLinkPagamento($parcela, $cliente, $master);
            
            $dados = [
                'nome' => $cliente->nome,
                'vencimento' => $parcela->datapagamento,
                'valor' => 'R$ ' . number_format($parcela->parcela, 2, ',', '.'),
                'link' => $link,
                'empresa' => $master->nomecom ?? $master->name,
                'cnpj' => $master->cnpj ?? '',
                'endereco' => $master->enderecom ?? '',
                'contato' => $master->contato ?? '',
            ];

            // Substitui variáveis na mensagem
            $mensagem = $whatsapp->replaceVariables($mensagemConfig->msg ?? '', $dados);

            // Adiciona mensagem à fila
            $whatsapp->queueMessage('text', $cliente->celular, $mensagem);

            // Envia QR Code PIX se configurado
            if ($master->msgpix == '1' && $parcela->gatewayPayment != 2) {
                $qrCode = $this->getQRCodePagamento($parcela, $master);
                if ($qrCode) {
                    $whatsapp->queueMessage(
                        'media', 
                        $cliente->celular, 
                        'Pague agora via PIX. Leia o QRCode.', 
                        $qrCode
                    );
                }
            }

            // Envia copia e cola se configurado
            if ($master->msgqr == '1' && $parcela->gatewayPayment != 2) {
                $copiaECola = $this->getCopiaECola($parcela, $master);
                if ($copiaECola) {
                    $whatsapp->queueMessage('text', $cliente->celular, $copiaECola);
                }
            }

            // Mensagem de aviso automático
            $avisoAutomatico = "*ATENÇÃO* Esta é uma mensagem automática e não precisa ser respondida.\n" .
                              "*Caso já tenha efetuado o pagamento por favor desconsidere esta cobrança.*";
            $whatsapp->queueMessage('text', $cliente->celular, $avisoAutomatico);

            // Processa fila
            $whatsapp->processQueue(10);

            return true;

        } catch (\Exception $e) {
            Log::error('Erro ao enviar cobrança', [
                'parcela_id' => $parcela->id,
                'cliente_id' => $cliente->id,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Gera link de pagamento
     */
    private function gerarLinkPagamento(Financeiro2 $parcela, Cliente $cliente, User $master): string
    {
        if ($master->pagamentos == 1) {
            // Mercado Pago
            $mp = MercadoPago::where('instancia', $parcela->id)->first();
            if ($mp) {
                return url("/pagamento?idCob={$mp->id}&idInst={$parcela->id}&idFin={$parcela->id}");
            }
        }

        return url('/');
    }

    /**
     * Obtém QR Code do pagamento
     */
    private function getQRCodePagamento(Financeiro2 $parcela, User $master): ?string
    {
        try {
            if ($master->pagamentos == 1) {
                // Mercado Pago
                $mp = MercadoPago::where('instancia', $parcela->id)
                    ->where('status', 'pending')
                    ->whereDate('data', '>=', now()->subDays(5))
                    ->first();
                
                return $mp->qrcode ?? null;
            }

            // Outros gateways (Gerencianet, Paghiper, Asaas)
            $finGlobal = FinanceiroGlobal::where('instancia', $parcela->id)->first();
            if ($finGlobal && $finGlobal->type === 'pix') {
                return $finGlobal->imagemQrcode;
            }

            return null;

        } catch (\Exception $e) {
            Log::error('Erro ao obter QR Code', [
                'parcela_id' => $parcela->id,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Obtém copia e cola do pagamento
     */
    private function getCopiaECola(Financeiro2 $parcela, User $master): ?string
    {
        try {
            if ($master->pagamentos == 1) {
                // Mercado Pago
                $mp = MercadoPago::where('instancia', $parcela->id)
                    ->where('status', 'pending')
                    ->whereDate('data', '>=', now()->subDays(5))
                    ->first();
                
                return $mp->linhad ?? null;
            }

            // Outros gateways
            $finGlobal = FinanceiroGlobal::where('instancia', $parcela->id)->first();
            if ($finGlobal) {
                if ($finGlobal->type === 'pix') {
                    return $finGlobal->copiacola;
                } elseif ($finGlobal->type === 'boleto') {
                    return "Código de barras:\n" . $finGlobal->codigobarra;
                }
            }

            return null;

        } catch (\Exception $e) {
            Log::error('Erro ao obter copia e cola', [
                'parcela_id' => $parcela->id,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }
}
