<?php

namespace App\Http\Controllers;

use App\Models\Carteira;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    /**
     * Exibe o formulário de login
     */
    public function showLoginForm()
    {
        $carteira = Carteira::where('id', 1)->first();
        
        return view('auth.login', [
            'nomesistema' => $carteira->nomecom ?? 'Gestor PRO',
            'favicon' => $carteira->favicon ?? '/img/favicon_default.png',
            'background' => $carteira->background ?? '',
        ]);
    }

    /**
     * Processa o login
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'login' => 'required',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Preencha todos os campos.',
            ]);
        }

        // Busca usuário
        $user = Carteira::where('login', $request->login)
                       ->where('status', '1')
                       ->first();

        if (!$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'E-mail ou Senha incorreto.',
            ]);
        }

        // Verifica senha (SHA1 do sistema antigo)
        if (sha1($request->password) !== $user->senha) {
            return response()->json([
                'status' => 'error',
                'message' => 'E-mail ou Senha incorreto.',
            ]);
        }

        // Faz login
        Auth::login($user);
        
        // Define redirecionamento baseado no tipo
        // Usa route() para gerar URL correta baseada na configuração do Laravel
        $redirect = route('dashboard');

        return response()->json([
            'status' => 'success',
            'message' => 'Login efetuado com sucesso',
            'redirect' => $redirect,
        ]);
    }

    /**
     * Logout
     */
    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/');
    }

    /**
     * Exibe formulário de criar conta
     */
    public function showRegisterForm()
    {
        $carteira = Carteira::where('id', 1)->first();
        
        return view('auth.register', [
            'nomesistema' => $carteira->nomecom ?? 'Gestor PRO',
            'favicon' => $carteira->favicon ?? '/img/favicon_default.png',
        ]);
    }

    /**
     * Processa cadastro
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nome' => 'required|max:120',
            'login' => 'required|email|unique:carteiras,login',
            'senha' => 'required|min:6|confirmed',
            'celular' => 'required',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Cria novo usuário
        $carteira = Carteira::create([
            'nome' => $request->nome,
            'login' => $request->login,
            'senha' => sha1($request->senha), // Mantém SHA1 por compatibilidade
            'celular' => $request->celular,
            'tipo' => '3', // Usuário comum
            'status' => '1',
            'idm' => '1',
            'assinatura' => now()->addDays(30)->format('d/m/Y'), // 30 dias trial
        ]);

        return redirect()->route('login')->with('success', 'Conta criada com sucesso! Faça login.');
    }

    /**
     * Exibe formulário de recuperar senha
     */
    public function showRecoverForm()
    {
        $carteira = Carteira::where('id', 1)->first();
        
        return view('auth.recover', [
            'nomesistema' => $carteira->nomecom ?? 'Gestor PRO',
            'favicon' => $carteira->favicon ?? '/img/favicon_default.png',
        ]);
    }

    /**
     * Processa recuperação de senha
     */
    public function recover(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $carteira = Carteira::where('login', $request->email)->first();

        if (!$carteira) {
            return back()->with('error', 'E-mail não encontrado.');
        }

        // TODO: Implementar envio de email com token de recuperação
        
        return back()->with('success', 'Instruções de recuperação enviadas para seu e-mail.');
    }
}

