<?php

namespace App\Http\Controllers;

use App\Models\Categoria;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class CategoriaController extends Controller
{
    /**
     * Lista todas as categorias
     */
    public function index()
    {
        $user = Auth::user();
        
        $categorias = Categoria::where('idu', $user->id)
            ->orderBy('nome', 'asc')
            ->get();

        $totalCategorias = $categorias->count();
        $categoriasProdutos = $categorias->filter(function ($categoria) {
            $tipo = $categoria->tipo ?? 'produto';
            return $tipo === 'produto' || $tipo === null;
        });
        $categoriasClientes = $categorias->filter(function ($categoria) {
            return ($categoria->tipo ?? 'produto') === 'cliente';
        });

        return view('categorias.index', compact('categorias', 'totalCategorias', 'categoriasProdutos', 'categoriasClientes'));
    }

    /**
     * Exibe formulário de criar categoria
     */
    public function create()
    {
        return view('categorias.create');
    }

    /**
     * Salva nova categoria
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'nome' => 'required|max:60',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        Categoria::create([
            'idu' => $user->id,
            'nome' => strtoupper($request->nome),
            'tipo' => $request->tipo ?? 'produto',
        ]);

        return redirect()->route('categorias.index')->with('success', 'Categoria cadastrada com sucesso!');
    }

    /**
     * Exibe formulário de edição
     */
    public function edit($id)
    {
        $user = Auth::user();
        
        $categoria = Categoria::where('id', $id)
            ->where('idu', $user->id)
            ->firstOrFail();

        return view('categorias.edit', compact('categoria'));
    }

    /**
     * Atualiza categoria
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();

        $categoria = Categoria::where('id', $id)
            ->where('idu', $user->id)
            ->firstOrFail();

        $validator = Validator::make($request->all(), [
            'nome' => 'required|max:60',
            'tipo' => 'nullable|in:produto,cliente',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $categoria->update([
            'nome' => strtoupper($request->nome),
            'tipo' => $request->tipo ?? $categoria->tipo ?? 'produto',
        ]);

        return redirect()->route('categorias.index')->with('success', 'Categoria cadastrada com sucesso!');
    }

    /**
     * Deleta categoria
     */
    public function destroy($id)
    {
        $user = Auth::user();

        $categoria = Categoria::where('id', $id)
            ->where('idu', $user->id)
            ->firstOrFail();

        // Verifica se tem clientes associados
        if ($categoria->clientes()->count() > 0) {
            return back()->with('error', 'Não é possível excluir categoria com clientes associados.');
        }

        $categoria->delete();

        return redirect()->route('categorias.index')->with('success', 'Categoria cadastrada com sucesso!');
    }
}

