<?php

namespace App\Http\Controllers;

use App\Models\Cliente;
use App\Models\Financeiro1;
use App\Models\Financeiro2;
use App\Models\Financeiro3;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Carbon\Carbon;

class FinanceiroController extends Controller
{
    /**
     * Lista contas a receber
     */
    public function contasReceber(Request $request)
    {
        $user = Auth::user();
        $mes = $request->get('mes', date('m'));
        $ano = date('Y');
        $datam = $mes . '/' . $ano;

        // SOMA VALORES A RECEBER
        $valoresAReceber = Financeiro2::where('status', '1')
            ->where('datapagamento', 'like', '%' . $datam . '%')
            ->where('idm', $user->id)
            ->sum('parcela');

        // SOMA VALORES RECEBIDOS
        $valoresRecebidos = Financeiro2::where('status', '2')
            ->where('pagoem', 'like', '%' . $datam . '%')
            ->where('idm', $user->id)
            ->sum('parcela');

        // EMPRÉSTIMOS ATIVOS
        $empativos = Financeiro1::where('status', '1')
            ->where('idm', $user->id)
            ->count();

        // PARCELAS ABERTAS
        $parcelasab = Financeiro2::where('status', '1')
            ->where('datapagamento', 'like', '%' . $datam . '%')
            ->where('idm', $user->id)
            ->count();

        // PARCELAS PAGAS
        $parcelasap = Financeiro2::where('status', '2')
            ->where('pagoem', 'like', '%' . $datam . '%')
            ->where('idm', $user->id)
            ->count();

        // CONTAS VENCIDAS
        $contasVencidas = Financeiro2::where('status', '1')
            ->where('idm', $user->id)
            ->whereRaw("STR_TO_DATE(datapagamento, '%d/%m/%Y') < CURDATE()")
            ->count();

        // VALOR TOTAL VENCIDO
        $valorVencido = Financeiro2::where('status', '1')
            ->where('idm', $user->id)
            ->whereRaw("STR_TO_DATE(datapagamento, '%d/%m/%Y') < CURDATE()")
            ->sum('parcela');

        // CLIENTES
        $cadcli = \App\Models\Cliente::where('idm', $user->id)->count();

        // Cobranças ativas (financeiro1 com status 1)
        $cobrancas = Financeiro1::where('status', '1')
            ->where('idm', $user->id)
            ->with(['cliente'])
            ->get();

        $meses = [
            '01' => 'Janeiro', '02' => 'Fevereiro', '03' => 'Março',
            '04' => 'Abril', '05' => 'Maio', '06' => 'Junho',
            '07' => 'Julho', '08' => 'Agosto', '09' => 'Setembro',
            '10' => 'Outubro', '11' => 'Novembro', '12' => 'Dezembro'
        ];

        $mesNome = $meses[$mes] ?? 'Mês Atual';

        // Tipos de cobranças
        $tiposCobrancas = [
            1 => 'Mercado Pago',
            2 => 'Sem Gateway',
            4 => 'PagHiper',
            5 => 'Asaas'
        ];

        return view('financeiro.contas-receber', compact(
            'cobrancas', 'mes', 'ano', 'meses', 'mesNome',
            'valoresAReceber', 'valoresRecebidos', 'empativos',
            'parcelasab', 'parcelasap', 'contasVencidas', 'valorVencido',
            'cadcli', 'tiposCobrancas'
        ));
    }

    /**
     * Formulário de cadastrar conta a receber
     */
    public function criarContaReceber()
    {
        $user = Auth::user();
        $clientes = Cliente::where('idm', $user->id)->orderBy('nome')->get();
        $categorias = \App\Models\Categoria::where('idu', $user->id)->orderBy('nome')->get();

        return view('financeiro.criar-conta-receber', compact('clientes', 'categorias'));
    }

    /**
     * Salva conta a receber
     */
    public function storeContaReceber(Request $request)
    {
        $user = Auth::user();
        
        $validator = Validator::make($request->all(), [
            'idc' => 'required|exists:clientes,id',
            'descricao' => 'required|max:255',
            'valor' => 'required',
            'parcelas' => 'required|integer|min:1',
            'vencimento' => 'required|date',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Converte valor brasileiro para decimal
        $valor = str_replace('.', '', $request->valor);
        $valor = str_replace(',', '.', $valor);
        $valor = floatval($valor);

        $chave = Str::random(60);
        $valorParcela = $valor / $request->parcelas;

        // Cria registro principal
        $financeiro1 = Financeiro1::create([
            'idc' => $request->idc,
            'idm' => $user->id,
            'valorsolicitado' => $valor,
            'valorfinal' => $valor,
            'parcelas' => $request->parcelas,
            'primeiraparcela' => Carbon::parse($request->vencimento)->format('d/m/Y'),
            'chave' => $chave,
            'vparcela' => $valorParcela,
            'gatewayPayment' => 1, // Default
            'status' => '1',
            'entrada' => Carbon::now()->format('d/m/Y'),
        ]);

        // Cria parcelas
        $dataParcela = Carbon::parse($request->vencimento);
        
        for ($i = 1; $i <= $request->parcelas; $i++) {
            Financeiro2::create([
                'idc' => $request->idc,
                'idm' => $user->id,
                'chave' => $chave,
                'parcela' => $valorParcela,
                'datapagamento' => $dataParcela->format('d/m/Y'),
                'status' => '1',
                'gatewayPayment' => 1,
            ]);

            $dataParcela->addMonth();
        }

        return redirect()->route('financeiro.contas-receber')->with('success', 'Conta a receber cadastrada com sucesso!');
    }

    /**
     * Baixar parcela (marcar como paga)
     */
    public function baixarParcela($id)
    {
        $user = Auth::user();
        
        $parcela = Financeiro2::where('id', $id)
            ->where('idm', $user->id)
            ->firstOrFail();

        $parcela->update([
            'status' => '2',
            'pagoem' => Carbon::now()->format('d/m/Y'),
        ]);

        // Verifica se todas as parcelas estão pagas
        $parcelasAbertas = Financeiro2::where('chave', $parcela->chave)
            ->where('status', '1')
            ->count();

        if ($parcelasAbertas == 0) {
            // Marca cobrança como quitada
            Financeiro1::where('chave', $parcela->chave)->update(['status' => '2']);
        }

        return back()->with('success', 'Operação realizada com sucesso!');
    }

    /**
     * Lista contas a pagar
     */
    public function contasPagar(Request $request)
    {
        $user = Auth::user();
        $mes = $request->get('mes', date('m'));
        $ano = date('Y');
        $datamFiltro = "$ano-$mes";

        // 💰 DESPESAS - Total do mês (pagas + não pagas)
        // datavencimento pode estar no formato Y-m-d ou d/m/Y
        $totalDespesas = Financeiro3::where('tipo', 'saida')
            ->where(function($query) use ($datamFiltro, $mes) {
                $query->where('datavencimento', 'like', "$datamFiltro%")
                      ->orWhere('datavencimento', 'like', "%/$mes/%");
            })
            ->where('idm', $user->id)
            ->sum('valor') ?? 0;

        // 💰 DESPESAS PAGAS (status = 1 = pago no legado)
        $despesasPagas = Financeiro3::where('tipo', 'saida')
            ->where('status', '1')
            ->where(function($query) use ($datamFiltro, $mes) {
                $query->where('datavencimento', 'like', "$datamFiltro%")
                      ->orWhere('datavencimento', 'like', "%/$mes/%");
            })
            ->where('idm', $user->id)
            ->sum('valor') ?? 0;

        // 💰 DESPESAS PENDENTES
        $despesasPendentes = $totalDespesas - $despesasPagas;

        // 🔴 Contas Vencidas (status = 0 = pendente, datavencimento < hoje)
        $vencidas = Financeiro3::where('tipo', 'saida')
            ->where('status', '0')
            ->where(function($query) use ($datamFiltro, $mes) {
                $query->where('datavencimento', 'like', "$datamFiltro%")
                      ->orWhere('datavencimento', 'like', "%/$mes/%");
            })
            ->where(function($query) {
                $query->whereRaw("STR_TO_DATE(datavencimento, '%Y-%m-%d') < CURDATE()")
                      ->orWhereRaw("STR_TO_DATE(datavencimento, '%d/%m/%Y') < CURDATE()");
            })
            ->where('idm', $user->id)
            ->selectRaw('COUNT(*) as total, SUM(valor) as valor_total')
            ->first();
        $totalVencidas = $vencidas->total ?? 0;
        $valorVencido = $vencidas->valor_total ?? 0;

        // ⏰ A Vencer esta Semana
        $aVencer = Financeiro3::where('tipo', 'saida')
            ->where('status', '0')
            ->whereBetween('datavencimento', [
                date('Y-m-d'),
                date('Y-m-d', strtotime('+7 days'))
            ])
            ->where('idm', $user->id)
            ->selectRaw('COUNT(*) as total, SUM(valor) as valor_total')
            ->first();
        $totalAVencer = $aVencer->total ?? 0;
        $valorAVencer = $aVencer->valor_total ?? 0;

        // 📊 Despesas por Categoria (se a tabela existir)
        $despesasPorCategoria = [];
        $categorias_labels = [];
        $categorias_valores = [];
        $categorias_cores = [];
        try {
            if (\Illuminate\Support\Facades\DB::getSchemaBuilder()->hasTable('categorias_financeiras')) {
                $despesasPorCategoria = \Illuminate\Support\Facades\DB::table('financeiro3 as f')
                    ->leftJoin('categorias_financeiras as c', 'f.categoria_id', '=', 'c.id')
                    ->where('f.tipo', 'saida')
                    ->where(function($q) use ($datamFiltro, $mes) {
                        $q->where('f.datavencimento', 'like', "$datamFiltro%")
                          ->orWhere('f.datavencimento', 'like', "%/$mes/%");
                    })
                    ->where('f.idm', $user->id)
                    ->selectRaw('c.nome as categoria, c.cor as cor, SUM(f.valor) as total')
                    ->groupBy('f.categoria_id', 'c.nome', 'c.cor')
                    ->orderByDesc('total')
                    ->get();

                $cor_map = [
                    'danger' => '#dc3545',
                    'warning' => '#ffc107',
                    'info' => '#0dcaf0',
                    'success' => '#00FF41',
                    'primary' => '#00FF41',
                    'secondary' => '#6c757d',
                    'lilac' => '#FF00FF'
                ];

                foreach ($despesasPorCategoria as $cat) {
                    $categorias_labels[] = $cat->categoria ?: 'Sem Categoria';
                    $categorias_valores[] = floatval($cat->total);
                    $categorias_cores[] = $cor_map[$cat->cor ?? 'secondary'] ?? '#6c757d';
                }
            }
        } catch (\Exception $e) {
            // Tabela não existe, continuar sem gráfico
        }

        // 🔍 Filtros
        $filtro_categoria = $request->get('categoria', '');
        $filtro_status = $request->get('status', '');
        $busca = $request->get('busca', '');

        // Lançamentos com filtros
        $query = Financeiro3::where('tipo', 'saida')
            ->where(function($q) use ($datamFiltro, $mes) {
                $q->where('datavencimento', 'like', "$datamFiltro%")
                  ->orWhere('datavencimento', 'like', "%/$mes/%");
            })
            ->where('idm', $user->id);

        if ($filtro_categoria) {
            $query->where('categoria_id', $filtro_categoria);
        }

        if ($filtro_status === 'pago') {
            $query->where('status', '1');
        } elseif ($filtro_status === 'aberto') {
            $query->where('status', '0')
                  ->whereRaw("STR_TO_DATE(datavencimento, '%Y-%m-%d') >= CURDATE()");
        } elseif ($filtro_status === 'vencido') {
            $query->where('status', '0')
                  ->whereRaw("STR_TO_DATE(datavencimento, '%Y-%m-%d') < CURDATE()");
        }

        if ($busca) {
            $query->where(function($q) use ($busca) {
                $q->where('descricao', 'like', "%$busca%")
                  ->orWhere('categoria', 'like', "%$busca%");
            });
        }

        $contas = $query->orderBy('datavencimento', 'asc')->get();

        // Buscar todas as categorias para o filtro
        $todasCategorias = [];
        try {
            if (\Illuminate\Support\Facades\DB::getSchemaBuilder()->hasTable('categorias_financeiras')) {
                $todasCategorias = \Illuminate\Support\Facades\DB::table('categorias_financeiras')
                    ->where('idm', $user->id)
                    ->orderBy('nome')
                    ->get();
            }
        } catch (\Exception $e) {
            // Tabela não existe
        }

        $meses = [
            '01' => 'Janeiro', '02' => 'Fevereiro', '03' => 'Março',
            '04' => 'Abril', '05' => 'Maio', '06' => 'Junho',
            '07' => 'Julho', '08' => 'Agosto', '09' => 'Setembro',
            '10' => 'Outubro', '11' => 'Novembro', '12' => 'Dezembro'
        ];

        return view('financeiro.contas-pagar', compact(
            'contas', 'mes', 'ano', 'meses', 'totalDespesas', 'despesasPagas',
            'despesasPendentes', 'totalVencidas', 'valorVencido', 'totalAVencer',
            'valorAVencer', 'categorias_labels', 'categorias_valores', 'categorias_cores',
            'filtro_categoria', 'filtro_status', 'busca', 'todasCategorias', 'despesasPorCategoria'
        ));
    }

    /**
     * Formulário de criar conta a pagar
     */
    public function criarContaPagar()
    {
        $user = Auth::user();
        $categorias = \App\Models\Categoria::where('idu', $user->id)->orderBy('nome')->get();

        return view('financeiro.criar-conta-pagar', compact('categorias'));
    }

    /**
     * Salva conta a pagar
     */
    public function storeContaPagar(Request $request)
    {
        $user = Auth::user();
        
        $validator = Validator::make($request->all(), [
            'descricao' => 'required|max:120',
            'valor' => 'required|numeric',
            'vencimento' => 'required|date',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        Financeiro3::create([
            'idm' => $user->id,
            'descricao' => $request->descricao,
            'valor' => $request->valor,
            'datavencimento' => Carbon::parse($request->vencimento)->format('d/m/Y'),
            'observacao' => $request->observacao,
            'status' => '1',
        ]);

        return redirect()->route('financeiro.contas-pagar')->with('sucesso', true);
    }

    /**
     * Baixar conta a pagar
     */
    public function baixarContaPagar($id)
    {
        $user = Auth::user();
        
        $conta = Financeiro3::where('id', $id)
            ->where('idm', $user->id)
            ->firstOrFail();

        $conta->update([
            'status' => '2',
            'datapagamento' => Carbon::now()->format('Y-m'),
        ]);

        return back()->with('success', 'Operação realizada com sucesso!');
    }

    /**
     * Contas Vencidas
     */
    public function contasVencidas(Request $request)
    {
        $user = Auth::user();

        // VALORES VENCIDOS
        $valoresVencidos = Financeiro2::where('status', '1')
            ->where('idm', $user->id)
            ->whereRaw("STR_TO_DATE(datapagamento, '%d/%m/%Y') < CURDATE()")
            ->sum('parcela');

        // CONTAS VENCIDAS
        $contasVencidas = Financeiro2::where('status', '1')
            ->where('idm', $user->id)
            ->whereRaw("STR_TO_DATE(datapagamento, '%d/%m/%Y') < CURDATE()")
            ->count();

        // CLIENTES COM CONTAS VENCIDAS
        $clientesVencidos = Financeiro2::where('status', '1')
            ->where('idm', $user->id)
            ->whereRaw("STR_TO_DATE(datapagamento, '%d/%m/%Y') < CURDATE()")
            ->distinct()
            ->count('idc');

        // Buscar cobranças com parcelas vencidas
        $cobrancasVencidas = Financeiro1::where('status', '1')
            ->where('idm', $user->id)
            ->with(['cliente'])
            ->get()
            ->filter(function($cobranca) use ($user) {
                return Financeiro2::where('chave', $cobranca->chave)
                    ->where('status', '1')
                    ->where('idm', $user->id)
                    ->whereRaw("STR_TO_DATE(datapagamento, '%d/%m/%Y') < CURDATE()")
                    ->exists();
            });

        // Tipos de cobranças
        $tiposCobrancas = [
            1 => 'Mercado Pago',
            2 => 'Sem Gateway',
            4 => 'PagHiper',
            5 => 'Asaas'
        ];

        return view('financeiro.contas-vencidas', compact(
            'cobrancasVencidas', 'valoresVencidos', 'contasVencidas',
            'clientesVencidos', 'tiposCobrancas'
        ));
    }

    /**
     * Histórico de transações
     */
    public function historico(Request $request)
    {
        $user = Auth::user();

        // Buscar transações finalizadas (status = 2)
        $transacoes = Financeiro2::where('status', '2')
            ->where('idm', $user->id)
            ->with(['financeiro1.cliente'])
            ->orderBy('pagoem', 'asc')
            ->get()
            ->map(function($transacao) {
                // Contar parcelas pagas
                $parcelasPagas = Financeiro2::where('chave', $transacao->chave)
                    ->where('status', '2')
                    ->count();
                
                $financeiro1 = $transacao->financeiro1;
                $cliente = $financeiro1 ? $financeiro1->cliente : null;
                
                return (object) [
                    'Id' => $financeiro1 ? ($financeiro1->id ?? $transacao->id) : $transacao->id,
                    'id' => $transacao->id,
                    'nome' => $cliente ? ($cliente->nome ?? '-') : '-',
                    'datapagamento' => $transacao->datapagamento ?? '-',
                    'pagoem' => $transacao->pagoem ?? '-',
                    'parcela' => $transacao->parcela ?? 0,
                    'parcelas' => $financeiro1 ? ($financeiro1->parcelas ?? 0) : 0,
                    'parcelas_pagas' => $parcelasPagas,
                    'status' => $transacao->status ?? '2',
                    'chave' => $transacao->chave ?? ''
                ];
            })
            ->filter(function($transacao) {
                // Filtrar apenas transações com dados válidos
                return !empty($transacao->nome) && $transacao->nome !== '-';
            })
            ->values(); // Reindexar array

        return view('financeiro.historico', compact('transacoes'));
    }

    /**
     * Extrato (contas quitadas)
     */
    public function extrato(Request $request)
    {
        $user = Auth::user();

        // Busca cobranças finalizadas (status = 2)
        $cobrancasFinalizadas = Financeiro1::where('status', '2')
            ->where('idm', $user->id)
            ->with(['cliente'])
            ->orderBy('created_at', 'desc')
            ->get();

        return view('financeiro.extrato', compact('cobrancasFinalizadas'));
    }

    /**
     * Simulador de parcelas (antes de criar)
     * Migrado do legado: /master/cad_contas_simulador.php
     */
    public function simularParcelas(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'cliente' => 'required|exists:clientes,id',
            'valor' => 'required',
            'parcelas' => 'required|integer|min:1|max:999',
            'datap' => 'required|date',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Converte valor
        $valor = str_replace('.', '', $request->valor);
        $valor = str_replace(',', '.', $valor);
        $valorParcela = floatval($valor);

        $cliente = Cliente::findOrFail($request->cliente);
        $parcelas = (int) $request->parcelas;
        $dataPrimeira = Carbon::parse($request->datap);

        // Calcula todas as parcelas
        $simulacao = [];
        for ($i = 0; $i < $parcelas; $i++) {
            $dataVencimento = clone $dataPrimeira;
            $dataVencimento->addMonths($i);
            
            // Ajusta se o dia não existe no mês (ex: 31/02 vira 28/02)
            if ($dataVencimento->day != $dataPrimeira->day) {
                $dataVencimento->modify('last day of previous month');
            }

            $simulacao[] = [
                'numero' => $i + 1,
                'valor' => $valorParcela,
                'vencimento' => $dataVencimento->format('d/m/Y'),
            ];
        }

        $dadosSimulacao = [
            'cliente' => $cliente,
            'valorTotal' => $valorParcela * $parcelas,
            'valorParcela' => $valorParcela,
            'parcelas' => $parcelas,
            'primeiraParcela' => $dataPrimeira->format('d/m/Y'),
            'primeiraParcelaOriginal' => $request->datap,
            'chave' => md5(uniqid(rand(), true)),
            'formapagamento' => '30', // Mensal
            'simulacao' => $simulacao,
        ];

        return view('financeiro.simulador', $dadosSimulacao);
    }

    /**
     * Confirma simulação e cria parcelas
     * Migrado do legado: /master/classes/simulador_exe.php
     */
    public function confirmarSimulacao(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'idcliente' => 'required|exists:clientes,id',
            'formapagamento' => 'required',
            'parcelas' => 'required|integer|min:1',
            'dataparcela' => 'required',
            'vparcela' => 'required|numeric',
            'typePayment' => 'required|in:pix,boleto',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        try {
            $chave = $request->idpedido ?? md5(uniqid(rand(), true));
            $dataParcela = Carbon::createFromFormat('d/m/Y', $request->dataparcela);
            
            // Cria financeiro1
            Financeiro1::create([
                'idc' => $request->idcliente,
                'idm' => $user->id,
                'valorfinal' => $request->vparcela,
                'formapagamento' => $request->formapagamento,
                'parcelas' => $request->parcelas,
                'primeiraparcela' => $request->dataparcelax ?? $request->dataparcela,
                'chave' => $chave,
                'vparcela' => $request->vparcela,
                'entrada' => now()->format('d/m/Y'),
                'gatewayPayment' => $user->pagamentos,
            ]);

            // Cria parcelas
            for ($i = 1; $i <= $request->parcelas; $i++) {
                $dataVenc = clone $dataParcela;
                $dataVenc->addMonths($i - 1);
                
                if ($dataVenc->day != $dataParcela->day) {
                    $dataVenc->modify('last day of previous month');
                }

                Financeiro2::create([
                    'idc' => $request->idcliente,
                    'chave' => $chave,
                    'idm' => $user->id,
                    'parcela' => $request->vparcela,
                    'datapagamento' => $dataVenc->format('d/m/Y'),
                    'gatewayPayment' => $user->pagamentos,
                ]);
            }

            return redirect()->route('financeiro.contas-receber')
                ->with('success', 'Cobrança cadastrada com sucesso!');

        } catch (\Exception $e) {
            \Log::error('Erro ao confirmar simulação', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return back()->with('error', 'Erro ao criar cobrança: ' . $e->getMessage())->withInput();
        }
    }
}

