<?php

namespace App\Http\Controllers;

use App\Models\Funcionario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class FuncionarioController extends Controller
{
    /**
     * Lista todos os funcionários
     */
    public function index()
    {
        $user = Auth::user();
        
        $funcionarios = Funcionario::where('id_saas', $user->id)
            ->orderBy('nome')
            ->get();

        return view('funcionarios.index', compact('funcionarios'));
    }

    /**
     * Mostra formulário de criação
     */
    public function create()
    {
        return view('funcionarios.create');
    }

    /**
     * Salva novo funcionário
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'nome' => 'required|string|max:255',
            'celular' => 'nullable|string|max:15',
            'login' => 'required|string|max:50|unique:funcionarios,login,NULL,id,id_saas,' . $user->id,
            'senha' => 'required|string|min:6|max:255',
            'roles' => 'required|in:funcionario',
            'porcentagem_vendas' => 'nullable|numeric|min:0|max:100',
            'funcoes' => 'nullable|array',
            'funcoes.*' => 'in:categorias,produtos,clientes,contas_receber,contas_pagar,finalizados,mensagens,pdv,estoque,vendas,comissoes',
        ], [
            'login.unique' => 'Este login já está em uso por outro funcionário.',
            'senha.min' => 'A senha deve ter no mínimo 6 caracteres.',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $funcionario = Funcionario::create([
            'id_saas' => $user->id,
            'nome' => $request->nome,
            'celular' => $request->celular,
            'login' => $request->login,
            'senha' => Hash::make($request->senha),
            'roles' => $request->roles,
            'porcentagem_vendas' => $request->porcentagem_vendas ?? 0,
            'status' => true,
            'funcoes' => $request->funcoes ? implode(',', $request->funcoes) : null,
        ]);

        return redirect()->route('funcionarios.index')
            ->with('success', 'Funcionário cadastrado com sucesso!');
    }

    /**
     * Mostra detalhes do funcionário
     */
    public function show(Funcionario $funcionario)
    {
        $user = Auth::user();

        if ($funcionario->id_saas !== $user->id) {
            abort(403);
        }

        // Carregar estatísticas de vendas e comissões
        $vendas = $funcionario->vendas()
            ->where('status', 'ativa')
            ->orderBy('data_venda', 'desc')
            ->limit(10)
            ->get();

        $totalVendas = $funcionario->vendas()
            ->where('status', 'ativa')
            ->count();

        $totalFaturado = $funcionario->vendas()
            ->where('status', 'ativa')
            ->sum('total');

        $totalComissao = $funcionario->vendas()
            ->where('status', 'ativa')
            ->sum('comissao');

        return view('funcionarios.show', compact('funcionario', 'vendas', 'totalVendas', 'totalFaturado', 'totalComissao'));
    }

    /**
     * Mostra formulário de edição
     */
    public function edit(Funcionario $funcionario)
    {
        $user = Auth::user();

        if ($funcionario->id_saas !== $user->id) {
            abort(403);
        }

        return view('funcionarios.edit', compact('funcionario'));
    }

    /**
     * Atualiza funcionário
     */
    public function update(Request $request, Funcionario $funcionario)
    {
        $user = Auth::user();

        if ($funcionario->id_saas !== $user->id) {
            abort(403);
        }

        $validator = Validator::make($request->all(), [
            'nome' => 'required|string|max:255',
            'celular' => 'nullable|string|max:15',
            'login' => 'required|string|max:50|unique:funcionarios,login,' . $funcionario->id . ',id,id_saas,' . $user->id,
            'senha' => 'nullable|string|min:6|max:255',
            'roles' => 'required|in:funcionario',
            'porcentagem_vendas' => 'nullable|numeric|min:0|max:100',
            'status' => 'boolean',
            'funcoes' => 'nullable|array',
            'funcoes.*' => 'in:categorias,produtos,clientes,contas_receber,contas_pagar,finalizados,mensagens,pdv,estoque,vendas,comissoes',
        ], [
            'login.unique' => 'Este login já está em uso por outro funcionário.',
            'senha.min' => 'A senha deve ter no mínimo 6 caracteres.',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $data = [
            'nome' => $request->nome,
            'celular' => $request->celular,
            'login' => $request->login,
            'roles' => $request->roles,
            'porcentagem_vendas' => $request->porcentagem_vendas ?? 0,
            'status' => $request->has('status') ? (bool)$request->status : $funcionario->status,
            'funcoes' => $request->funcoes ? implode(',', $request->funcoes) : null,
        ];

        if ($request->filled('senha')) {
            $data['senha'] = Hash::make($request->senha);
        }

        $funcionario->update($data);

        return redirect()->route('funcionarios.index')
            ->with('success', 'Funcionário atualizado com sucesso!');
    }

    /**
     * Remove funcionário
     */
    public function destroy(Funcionario $funcionario)
    {
        $user = Auth::user();

        if ($funcionario->id_saas !== $user->id) {
            abort(403);
        }

        // Verificar se há vendas associadas
        if ($funcionario->vendas()->count() > 0) {
            return back()->withErrors(['error' => 'Não é possível excluir funcionário com vendas associadas!']);
        }

        $funcionario->delete();

        return redirect()->route('funcionarios.index')
            ->with('success', 'Funcionário excluído com sucesso!');
    }
}

