<?php

namespace App\Http\Controllers;

use App\Models\Mensagem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class MensagemController extends Controller
{
    /**
     * Lista todas as mensagens
     */
    public function index()
    {
        $user = Auth::user();
        
        $mensagens = Mensagem::where('idu', $user->id)
            ->orderBy('created_at', 'desc')
            ->get();

        return view('mensagens.index', compact('mensagens'));
    }

    /**
     * Formulário de criar mensagem
     */
    public function create()
    {
        return view('mensagens.create');
    }

    /**
     * Salva nova mensagem
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'tipo' => 'required|in:1,2,3,4,5,6,7',
            'msg' => 'required',
            'hora' => ['required', 'regex:/^([01][0-9]|2[0-3]):[0-5][0-9]$/'],
            'status' => 'required|in:0,1',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        Mensagem::create([
            'idu' => $user->id,
            'msg' => $request->msg,
            'tipo' => $request->tipo,
            'status' => $request->status,
            'hora' => $request->hora,
        ]);

        return redirect()->route('mensagens.index')->with('success', 'Mensagem cadastrada com sucesso!');
    }

    /**
     * Formulário de edição
     */
    public function edit($id)
    {
        $user = Auth::user();
        
        $mensagem = Mensagem::where('id', $id)
            ->where('idu', $user->id)
            ->firstOrFail();

        return view('mensagens.edit', compact('mensagem'));
    }

    /**
     * Atualiza mensagem
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();

        $mensagem = Mensagem::where('id', $id)
            ->where('idu', $user->id)
            ->firstOrFail();

        $validator = Validator::make($request->all(), [
            'tipo' => 'required|in:1,2,3,4,5,6,7',
            'msg' => 'required',
            'hora' => ['required', 'regex:/^([01][0-9]|2[0-3]):[0-5][0-9]$/'],
            'status' => 'required|in:0,1',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $mensagem->update([
            'msg' => $request->msg,
            'tipo' => $request->tipo,
            'status' => $request->status,
            'hora' => $request->hora,
        ]);

        return redirect()->route('mensagens.index')->with('success', 'Mensagem atualizada com sucesso!');
    }

    /**
     * Deleta mensagem
     */
    public function destroy($id)
    {
        $user = Auth::user();

        $mensagem = Mensagem::where('id', $id)
            ->where('idu', $user->id)
            ->firstOrFail();

        $mensagem->delete();

        return redirect()->route('mensagens.index')->with('success', 'Mensagem excluída com sucesso!');
    }
}

