<?php

namespace App\Http\Controllers\Pdv;

use App\Http\Controllers\Controller;
use App\Models\PdvCaixa;
use App\Models\PdvVenda;
use App\Models\PdvFormaPagamento;
use App\Models\Funcionario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PdvController extends Controller
{
    /**
     * Tela principal do PDV
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $idm = $user->id;

        // Buscar caixa aberto
        $caixa = PdvCaixa::where('idm', $idm)
            ->aberto()
            ->first();

        // Calcular vendas de hoje
        $vendas_hoje = 0;
        $caixa_aberto = false;
        $data_caixa = null;
        $nome_atendente = null;
        $caixa_aberto_id = null;

        if ($caixa) {
            $caixa_aberto_id = $caixa->id;
            $caixa_aberto = true;
            $data_caixa = $caixa->abertura->format('d/m/Y H:i');
            $nome_atendente = $caixa->atendente_nome;

            // Buscar vendas ativas do caixa atual
            $vendas_hoje = PdvVenda::where('caixa_id', $caixa_aberto_id)
                ->where(function($query) {
                    $query->where('status', 'ativa')
                          ->orWhereNull('status');
                })
                ->sum('total');
        }

        // Buscar formas de pagamento
        $formas_pagamento = PdvFormaPagamento::where('idm', $idm)
            ->orderBy('nome')
            ->get();

        // Buscar funcionários para seleção de atendente
        $funcionarios = Funcionario::where('id_saas', $idm)
            ->where('status', true)
            ->orderBy('nome')
            ->get();

        // Buscar ciclos de pagamento (se a tabela existir)
        $ciclos_pagamento = [];
        try {
            if (DB::getSchemaBuilder()->hasTable('_pdv_ciclos_pagamento')) {
                $ciclos_pagamento = DB::table('_pdv_ciclos_pagamento')
                    ->where('ativo', 1)
                    ->orderBy('dias')
                    ->get();
            }
        } catch (\Exception $e) {
            // Tabela não existe, continuar sem ciclos
        }

        // Verificar código tempo real (se a tabela existir)
        $codigo_tempo_real_ativo = false;
        $codigo_tempo_real_numero = null;
        $codigo_tempo_real_data = null;

        try {
            if (DB::getSchemaBuilder()->hasTable('_pdv_app_codigos')) {
                $codigoAtivo = DB::table('_pdv_app_codigos')
                    ->where('idm', $idm)
                    ->where('ativo', 1)
                    ->orderBy('data_criacao', 'desc')
                    ->first();

                if ($codigoAtivo) {
                    $codigo_tempo_real_ativo = true;
                    $codigo_tempo_real_numero = $codigoAtivo->codigo;
                    $codigo_tempo_real_data = \Carbon\Carbon::parse($codigoAtivo->data_criacao)->format('d/m/Y H:i');
                }
            }
        } catch (\Exception $e) {
            // Tabela não existe, continuar sem código tempo real
        }

        // Lógica para repetir pedido (se houver parâmetros)
        $repetir_pedidos_itens = [];
        $cliente_repetir = null;
        if ($request->has('repetir_pedido') && $request->has('id_cliente')) {
            $repetir_pedido_ids = array_filter(array_map('intval', explode(',', $request->repetir_pedido)));
            $id_cliente = intval($request->id_cliente);

            // Buscar dados do cliente
            $cliente_repetir = \App\Models\Cliente::where('Id', $id_cliente)
                ->where('idm', $idm)
                ->first();

            // Buscar itens dos pedidos
            if ($cliente_repetir && count($repetir_pedido_ids) > 0) {
                $produtos_carrinho = [];
                foreach ($repetir_pedido_ids as $id_venda) {
                    $itens = DB::table('_pdv_itens as i')
                        ->join('_pdv_produtos as p', 'i.id_produto', '=', 'p.id')
                        ->where('i.id_venda', $id_venda)
                        ->select('i.id_produto', 'i.quantidade', 'i.preco_unitario', 'p.nome as produto_nome', 'p.codigo_barras')
                        ->get();

                    foreach ($itens as $item) {
                        $pid = $item->id_produto;
                        if (!isset($produtos_carrinho[$pid])) {
                            $produtos_carrinho[$pid] = [
                                'id' => $pid,
                                'nome' => $item->produto_nome,
                                'quantidade' => 0,
                                'preco' => floatval($item->preco_unitario),
                                'codigo_barras' => $item->codigo_barras
                            ];
                        }
                        $produtos_carrinho[$pid]['quantidade'] += intval($item->quantidade);
                    }
                }
                $repetir_pedidos_itens = array_values($produtos_carrinho);
            }
        }

        return view('pdv.index', compact(
            'caixa',
            'vendas_hoje',
            'caixa_aberto',
            'data_caixa',
            'nome_atendente',
            'caixa_aberto_id',
            'formas_pagamento',
            'funcionarios',
            'ciclos_pagamento',
            'codigo_tempo_real_ativo',
            'codigo_tempo_real_numero',
            'codigo_tempo_real_data',
            'repetir_pedidos_itens',
            'cliente_repetir'
        ));
    }
}

