<?php

namespace App\Http\Controllers\Pdv;

use App\Http\Controllers\Controller;
use App\Models\PdvProduto;
use App\Models\Fornecedor;
use App\Models\Categoria;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ProdutoController extends Controller
{
    /**
     * Lista todos os produtos
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        $query = PdvProduto::where('idm', $user->id)
            ->with(['fornecedor', 'categoria']);

        // Filtros
        if ($request->has('busca') && $request->busca) {
            $query->where(function($q) use ($request) {
                $q->where('nome', 'like', '%' . $request->busca . '%')
                  ->orWhere('codigo_barras', 'like', '%' . $request->busca . '%');
            });
        }

        if ($request->has('categoria_id') && $request->categoria_id) {
            $query->where('categoria_id', $request->categoria_id);
        }

        if ($request->has('estoque_baixo') && $request->estoque_baixo) {
            $query->estoqueBaixo();
        }

        $produtos = $query->orderBy('nome')->paginate(20);
        
        $categorias = Categoria::where('idu', $user->id)
            ->where(function($q) {
                $q->where('tipo', 'produto')->orWhereNull('tipo');
            })
            ->orderBy('nome')
            ->get();

        return view('pdv.produtos.index', compact('produtos', 'categorias'));
    }

    /**
     * Mostra formulário de criação
     */
    public function create()
    {
        $user = Auth::user();
        
        $fornecedores = Fornecedor::where('idm', $user->id)->orderBy('nome')->get();
        $categorias = Categoria::where('idu', $user->id)
            ->where(function($q) {
                $q->where('tipo', 'produto')->orWhereNull('tipo');
            })
            ->orderBy('nome')
            ->get();

        return view('pdv.produtos.create', compact('fornecedores', 'categorias'));
    }

    /**
     * Salva novo produto
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'nome' => 'required|string|max:255',
            'descricao' => 'nullable|string',
            'preco_venda' => 'required|numeric|min:0',
            'preco_custo' => 'nullable|numeric|min:0',
            'estoque' => 'required|integer|min:0',
            'estoque_minimo' => 'nullable|integer|min:0',
            'codigo_barras' => 'nullable|string|max:50|unique:_pdv_produtos,codigo_barras,NULL,id,idm,' . $user->id,
            'unidade' => 'nullable|string|max:10',
            'fabricante' => 'nullable|string|max:255',
            'peso' => 'nullable|numeric|min:0',
            'ncm' => 'nullable|string|max:20',
            'cfop' => 'nullable|string|max:10',
            'fornecedor_id' => 'nullable|exists:fornecedores,id',
            'categoria_id' => 'nullable|exists:categorias,id',
            'imagem' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $data = $request->all();
        $data['idm'] = $user->id;
        $data['alerta_estoque'] = $request->has('alerta_estoque') ? 1 : 0;

        // Upload de imagem
        if ($request->hasFile('imagem')) {
            $path = $request->file('imagem')->store('produtos', 'public');
            $data['imagem'] = '/storage/' . $path;
        }

        // Calcular margem de lucro se preço de custo for informado
        if ($request->preco_custo > 0 && $request->preco_venda > 0) {
            $data['margem_lucro'] = (($request->preco_venda - $request->preco_custo) / $request->preco_custo) * 100;
        }

        PdvProduto::create($data);

        return redirect()->route('pdv.produtos.index')
            ->with('success', 'Produto cadastrado com sucesso!');
    }

    /**
     * Mostra formulário de edição
     */
    public function edit(PdvProduto $produto)
    {
        $user = Auth::user();

        // Verificar se o produto pertence ao usuário
        if ($produto->idm !== $user->id) {
            abort(403);
        }

        $fornecedores = Fornecedor::where('idm', $user->id)->orderBy('nome')->get();
        $categorias = Categoria::where('idu', $user->id)
            ->where(function($q) {
                $q->where('tipo', 'produto')->orWhereNull('tipo');
            })
            ->orderBy('nome')
            ->get();

        return view('pdv.produtos.edit', compact('produto', 'fornecedores', 'categorias'));
    }

    /**
     * Atualiza produto
     */
    public function update(Request $request, PdvProduto $produto)
    {
        $user = Auth::user();

        // Verificar se o produto pertence ao usuário
        if ($produto->idm !== $user->id) {
            abort(403);
        }

        $request->validate([
            'nome' => 'required|string|max:255',
            'descricao' => 'nullable|string',
            'preco_venda' => 'required|numeric|min:0',
            'preco_custo' => 'nullable|numeric|min:0',
            'estoque' => 'required|integer|min:0',
            'estoque_minimo' => 'nullable|integer|min:0',
            'codigo_barras' => 'nullable|string|max:50|unique:_pdv_produtos,codigo_barras,' . $produto->id . ',id,idm,' . $user->id,
            'unidade' => 'nullable|string|max:10',
            'fabricante' => 'nullable|string|max:255',
            'peso' => 'nullable|numeric|min:0',
            'ncm' => 'nullable|string|max:20',
            'cfop' => 'nullable|string|max:10',
            'fornecedor_id' => 'nullable|exists:fornecedores,id',
            'categoria_id' => 'nullable|exists:categorias,id',
            'imagem' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $data = $request->all();
        $data['alerta_estoque'] = $request->has('alerta_estoque') ? 1 : 0;

        // Upload de nova imagem
        if ($request->hasFile('imagem')) {
            // Deletar imagem antiga se existir
            if ($produto->imagem) {
                $oldPath = str_replace('/storage/', '', $produto->imagem);
                Storage::disk('public')->delete($oldPath);
            }
            
            $path = $request->file('imagem')->store('produtos', 'public');
            $data['imagem'] = '/storage/' . $path;
        }

        // Calcular margem de lucro
        if ($request->preco_custo > 0 && $request->preco_venda > 0) {
            $data['margem_lucro'] = (($request->preco_venda - $request->preco_custo) / $request->preco_custo) * 100;
        }

        $produto->update($data);

        return redirect()->route('pdv.produtos.index')
            ->with('success', 'Produto atualizado com sucesso!');
    }

    /**
     * Remove produto
     */
    public function destroy(PdvProduto $produto)
    {
        $user = Auth::user();

        // Verificar se o produto pertence ao usuário
        if ($produto->idm !== $user->id) {
            abort(403);
        }

        // Deletar imagem se existir
        if ($produto->imagem) {
            $oldPath = str_replace('/storage/', '', $produto->imagem);
            Storage::disk('public')->delete($oldPath);
        }

        $produto->delete();

        return redirect()->route('pdv.produtos.index')
            ->with('success', 'Produto excluído com sucesso!');
    }

    /**
     * Busca produto por código de barras (API)
     */
    public function buscarPorCodigo(Request $request)
    {
        $user = Auth::user();
        $codigo = $request->codigo;

        $produto = PdvProduto::where('idm', $user->id)
            ->porCodigoBarras($codigo)
            ->first();

        if ($produto) {
            return response()->json([
                'success' => true,
                'produto' => [
                    'id' => $produto->id,
                    'nome' => $produto->nome,
                    'preco_venda' => $produto->preco_venda,
                    'estoque' => $produto->estoque,
                    'imagem' => $produto->imagem,
                ]
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Produto não encontrado'
        ], 404);
    }
}

