<?php

namespace App\Http\Controllers\Pdv;

use App\Http\Controllers\Controller;
use App\Models\PdvVenda;
use App\Models\PdvCaixa;
use App\Models\PdvProduto;
use App\Models\Cliente;
use App\Models\PdvFormaPagamento;
use App\Models\Funcionario;
use App\Services\PdvService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class VendaController extends Controller
{
    protected $pdvService;

    public function __construct(PdvService $pdvService)
    {
        $this->pdvService = $pdvService;
    }

    /**
     * Lista todas as vendas
     */
    public function index(Request $request)
    {
        $user = Auth::user();

        $query = PdvVenda::where('idm', $user->id)
            ->with(['cliente', 'funcionario', 'caixa', 'itens.produto']);

        // Filtros
        if ($request->has('data_inicio') && $request->data_inicio) {
            $query->whereDate('data_venda', '>=', $request->data_inicio);
        }

        if ($request->has('data_fim') && $request->data_fim) {
            $query->whereDate('data_venda', '<=', $request->data_fim);
        }

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        $vendas = $query->orderBy('data_venda', 'desc')->paginate(20);

        return view('pdv.vendas.index', compact('vendas'));
    }

    /**
     * Mostra a tela de criar venda (carrinho)
     */
    public function create()
    {
        $user = Auth::user();

        // Verificar se há caixa aberto
        $caixa = PdvCaixa::where('idm', $user->id)
            ->aberto()
            ->first();

        if (!$caixa) {
            return redirect()->route('pdv.index')
                ->withErrors(['error' => 'É necessário abrir um caixa antes de realizar vendas!']);
        }

        $formasPagamento = PdvFormaPagamento::where('idm', $user->id)->orderBy('nome')->get();

        return view('pdv.vendas.create', compact('caixa', 'formasPagamento'));
    }

    /**
     * Salva uma nova venda
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'caixa_id' => 'required|exists:_pdv_caixa,id',
            'cliente_id' => 'nullable|exists:clientes,Id',
            'funcionario_id' => 'nullable|exists:funcionarios,id',
            'forma_pagamento' => 'required|string',
            'itens' => 'required|array|min:1',
            'itens.*.produto_id' => 'required|exists:_pdv_produtos,id',
            'itens.*.quantidade' => 'required|integer|min:1',
            'itens.*.preco' => 'required|numeric|min:0',
            'total' => 'required|numeric|min:0',
            'desconto' => 'nullable|numeric|min:0',
            'lancar_conta' => 'nullable|boolean',
            'parcelas' => 'nullable|integer|min:1|max:12',
            'data_vencimento' => 'nullable|date',
        ]);

        // Verificar se o caixa pertence ao usuário e está aberto
        $caixa = PdvCaixa::where('idm', $user->id)
            ->where('id', $request->caixa_id)
            ->aberto()
            ->firstOrFail();

        // Verificar estoque de todos os produtos
        foreach ($request->itens as $item) {
            $produto = PdvProduto::findOrFail($item['produto_id']);
            
            if ($produto->idm !== $user->id) {
                return back()->withErrors(['error' => 'Produto não pertence ao seu cadastro!']);
            }

            if ($produto->estoque < $item['quantidade']) {
                return back()->withErrors(['error' => "Estoque insuficiente para o produto: {$produto->nome}"]);
            }
        }

        try {
            // Buscar funcionário pelo nome do atendente do caixa
            $funcionarioId = null;
            if ($caixa->atendente_nome) {
                $funcionario = Funcionario::where('id_saas', $user->id)
                    ->where('nome', $caixa->atendente_nome)
                    ->first();
                if ($funcionario) {
                    $funcionarioId = $funcionario->id;
                }
            }

            $dados = [
                'idm' => $user->id,
                'caixa_id' => $caixa->id,
                'cliente_id' => $request->cliente_id,
                'funcionario_id' => $funcionarioId,
                'total' => $request->total,
                'desconto' => $request->desconto ?? 0,
                'forma_pagamento' => $request->forma_pagamento,
                'itens' => $request->itens,
            ];

            $venda = $this->pdvService->criarVenda($dados);

            // Se deve lançar como conta a receber
            if ($request->lancar_conta && $request->cliente_id) {
                $this->lancarContaReceber($venda, $request);
            }

            return redirect()->route('pdv.vendas.show', $venda->id)
                ->with('success', 'Venda realizada com sucesso!');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Mostra detalhes de uma venda
     */
    public function show(PdvVenda $venda)
    {
        $user = Auth::user();

        if ($venda->idm !== $user->id) {
            abort(403);
        }

        $venda->load(['cliente', 'funcionario', 'caixa', 'itens.produto']);

        return view('pdv.vendas.show', compact('venda'));
    }

    /**
     * Cancela uma venda
     */
    public function cancelar(Request $request, PdvVenda $venda)
    {
        $user = Auth::user();

        if ($venda->idm !== $user->id) {
            abort(403);
        }

        try {
            $this->pdvService->cancelarVenda($venda);

            return redirect()->route('pdv.vendas.index')
                ->with('success', 'Venda cancelada com sucesso!');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => $e->getMessage()]);
        }
    }

    /**
     * Busca clientes (API)
     */
    public function buscarClientes(Request $request)
    {
        $user = Auth::user();
        $termo = $request->termo;

        $clientes = Cliente::where('idm', $user->id)
            ->where(function($q) use ($termo) {
                $q->where('nome', 'like', "%{$termo}%")
                  ->orWhere('cpf', 'like', "%{$termo}%")
                  ->orWhere('email', 'like', "%{$termo}%");
            })
            ->limit(10)
            ->get();

        return response()->json($clientes->map(function($cliente) {
            return [
                'id' => $cliente->Id,
                'nome' => $cliente->nome,
                'cpf' => $cliente->cpf,
                'email' => $cliente->email,
                'celular' => $cliente->celular,
            ];
        }));
    }

    /**
     * Busca produtos (API)
     */
    public function buscarProdutos(Request $request)
    {
        $user = Auth::user();
        $termo = $request->termo;

        $produtos = PdvProduto::where('idm', $user->id)
            ->where(function($q) use ($termo) {
                $q->where('nome', 'like', "%{$termo}%")
                  ->orWhere('codigo_barras', 'like', "%{$termo}%");
            })
            ->where('estoque', '>', 0)
            ->limit(20)
            ->get();

        return response()->json($produtos->map(function($produto) {
            return [
                'id' => $produto->id,
                'nome' => $produto->nome,
                'preco_venda' => $produto->preco_venda,
                'estoque' => $produto->estoque,
                'codigo_barras' => $produto->codigo_barras,
                'imagem' => $produto->imagem ? asset($produto->imagem) : null,
            ];
        }));
    }

    /**
     * Lança venda como conta a receber
     */
    private function lancarContaReceber(PdvVenda $venda, Request $request)
    {
        if (!$venda->cliente_id) {
            return;
        }

        $parcelas = $request->parcelas ?? 1;
        $valorParcela = $venda->total / $parcelas;
        $dataVencimento = $request->data_vencimento 
            ? \Carbon\Carbon::parse($request->data_vencimento)
            : now()->addMonth();

        for ($i = 0; $i < $parcelas; $i++) {
            DB::table('financeiro1')->insert([
                'idm' => $venda->idm,
                'idc' => $venda->cliente_id,
                'valor' => $valorParcela,
                'datapagamento' => $dataVencimento->copy()->addMonths($i)->format('d/m/Y'),
                'status' => '1',
                'descricao' => "Venda PDV #{$venda->id} - Parcela " . ($i + 1) . "/{$parcelas}",
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
    }
}

