<?php

namespace App\Http\Controllers;

use App\Models\Plano;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PlanoController extends Controller
{
    /**
     * Lista planos disponíveis
     */
    public function index()
    {
        $user = Auth::user();
        $planos = Plano::select('id', 'nome', 'valor', 'cadastros_lim', 'instancias_limite')->get();
        
        // Buscar dados do admin para validar token MP
        $dadosAdmin = \App\Models\Carteira::where(function($query) {
                $query->where('roles', 'superadmin')
                      ->orWhere('tipo', '1');
            })
            ->orWhere('tipo', '1')
            ->orderBy('id', 'asc')
            ->first();
        
        // Saldo disponível (sistema de indicações)
        $saldoDisponivel = 0;
        if ($user && $user->id) {
            try {
                if (DB::getSchemaBuilder()->hasTable('indicacoes_saldo')) {
                    $saldoData = DB::table('indicacoes_saldo')
                        ->where('id_usuario', $user->id)
                        ->first();
                    $saldoDisponivel = $saldoData ? floatval($saldoData->saldo_disponivel ?? 0) : 0;
                }
            } catch (\Exception $e) {
                // Tabela não existe, usar valor padrão
                $saldoDisponivel = 0;
            }
        }
        
        // Validar token MP
        $tem_mp = false;
        if ($dadosAdmin && !empty($dadosAdmin->tokenmp) && !empty($dadosAdmin->publicmp)) {
            // TODO: Implementar validação real do token MP
            $tem_mp = true; // Por enquanto, assumir que está válido se existir
        }

        return view('planos.index', compact('planos', 'user', 'dadosAdmin', 'saldoDisponivel', 'tem_mp'));
    }

    /**
     * Processa compra de plano
     */
    public function comprar(Request $request, $id)
    {
        $user = Auth::user();
        $plano = Plano::findOrFail($id);

        // TODO: Integrar com gateway de pagamento
        // Por enquanto, apenas redireciona

        return view('planos.comprar', compact('plano', 'user'));
    }

    /**
     * Admin: Lista planos
     */
    public function admin()
    {
        if (Auth::user()->tipo != '1') {
            abort(403);
        }

        $planos = Plano::all();
        return view('planos.admin', compact('planos'));
    }

    /**
     * Admin: Criar plano
     */
    public function store(Request $request)
    {
        if (Auth::user()->tipo != '1') {
            abort(403);
        }

        $request->validate([
            'nome' => 'required|max:255',
            'valor' => 'required|numeric|min:0',
        ]);

        Plano::create([
            'nome' => $request->nome,
            'valor' => $request->valor,
            'descricao' => $request->descricao,
            'ativo' => $request->has('ativo') ? '1' : '0',
            'destaque' => $request->has('destaque') ? '1' : '0',
        ]);

        return back()->with('success', 'Plano criado com sucesso!');
    }

    /**
     * Admin: Atualizar plano
     */
    public function update(Request $request, $id)
    {
        if (Auth::user()->tipo != '1') {
            abort(403);
        }

        $plano = Plano::findOrFail($id);

        $request->validate([
            'nome' => 'required|max:255',
            'valor' => 'required|numeric|min:0',
        ]);

        $plano->update([
            'nome' => $request->nome,
            'valor' => $request->valor,
            'descricao' => $request->descricao,
            'ativo' => $request->has('ativo') ? '1' : '0',
            'destaque' => $request->has('destaque') ? '1' : '0',
        ]);

        return back()->with('success', 'Plano atualizado com sucesso!');
    }

    /**
     * Admin: Deletar plano
     */
    public function destroy($id)
    {
        if (Auth::user()->tipo != '1') {
            abort(403);
        }

        $plano = Plano::findOrFail($id);
        $plano->delete();

        return back()->with('success', 'Plano excluído com sucesso!');
    }
}

