<?php

namespace App\Http\Controllers;

use App\Models\WhatsappEnvio;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class RelatorioWhatsAppController extends Controller
{
    /**
     * Exibe o relatório de envios WhatsApp
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $idm = $user->id;

        // Filtros de data inteligentes
        $periodo = $request->get('periodo', 'mes_atual');
        $dataInicio = $request->get('data_inicio', '');
        $dataFim = $request->get('data_fim', '');
        $gatewayFiltro = $request->get('gateway', '');

        // Processar período selecionado
        [$dataInicio, $dataFim] = $this->processarPeriodo($periodo, $dataInicio, $dataFim);

        // Construir query base
        $query = WhatsappEnvio::where('idm', $idm)
            ->whereBetween(DB::raw('DATE(data_envio)'), [$dataInicio, $dataFim]);

        // Filtro por gateway
        if (!empty($gatewayFiltro) && $gatewayFiltro !== 'todos') {
            $query->where('gateway', $gatewayFiltro);
        }

        // Estatísticas gerais
        $totalEnvios = (clone $query)->count();
        $enviosSucesso = (clone $query)->where('status', 'sucesso')->count();
        $enviosFalha = (clone $query)->where('status', 'falha')->count();
        $enviosPendentes = (clone $query)->where('status', 'pendente')->count();
        $taxaSucesso = $totalEnvios > 0 ? ($enviosSucesso / $totalEnvios) * 100 : 0;

        // Envios por tipo
        $enviosPorTipo = (clone $query)
            ->select('tipo', DB::raw('COUNT(*) as total'))
            ->groupBy('tipo')
            ->get();

        // Envios por gateway
        $enviosPorGateway = (clone $query)
            ->whereNotNull('gateway')
            ->select('gateway', DB::raw('COUNT(*) as total'))
            ->groupBy('gateway')
            ->get();

        // Envios por dia (últimos 7 dias)
        $seteDiasAtras = Carbon::now()->subDays(7)->format('Y-m-d');
        $enviosPorDia = WhatsappEnvio::where('idm', $idm)
            ->whereBetween(DB::raw('DATE(data_envio)'), [$seteDiasAtras, Carbon::now()->format('Y-m-d')])
            ->when(!empty($gatewayFiltro) && $gatewayFiltro !== 'todos', function($q) use ($gatewayFiltro) {
                return $q->where('gateway', $gatewayFiltro);
            })
            ->select(
                DB::raw('DATE(data_envio) as dia'),
                DB::raw('COUNT(*) as total'),
                DB::raw('SUM(CASE WHEN status = "sucesso" THEN 1 ELSE 0 END) as sucessos'),
                DB::raw('SUM(CASE WHEN status = "falha" THEN 1 ELSE 0 END) as falhas')
            )
            ->groupBy(DB::raw('DATE(data_envio)'))
            ->orderBy('dia', 'ASC')
            ->get();

        // Histórico detalhado (limitado a 1000 registros)
        $historico = (clone $query)
            ->orderBy('data_envio', 'DESC')
            ->limit(1000)
            ->get();

        // Calcular dias do período
        $diasPeriodo = Carbon::parse($dataInicio)->diffInDays(Carbon::parse($dataFim)) + 1;

        return view('relatorios.whatsapp', compact(
            'periodo',
            'dataInicio',
            'dataFim',
            'gatewayFiltro',
            'totalEnvios',
            'enviosSucesso',
            'enviosFalha',
            'enviosPendentes',
            'taxaSucesso',
            'enviosPorTipo',
            'enviosPorGateway',
            'enviosPorDia',
            'historico',
            'diasPeriodo'
        ));
    }

    /**
     * Processa o período selecionado e retorna as datas
     */
    private function processarPeriodo($periodo, $dataInicio, $dataFim): array
    {
        switch($periodo) {
            case 'hoje':
                $dataInicio = $dataFim = Carbon::today()->format('Y-m-d');
                break;
            case 'ontem':
                $dataInicio = $dataFim = Carbon::yesterday()->format('Y-m-d');
                break;
            case 'ultima_semana':
                $dataInicio = Carbon::now()->subDays(7)->format('Y-m-d');
                $dataFim = Carbon::today()->format('Y-m-d');
                break;
            case 'ultimo_mes':
                $dataInicio = Carbon::now()->subMonth()->startOfMonth()->format('Y-m-d');
                $dataFim = Carbon::now()->subMonth()->endOfMonth()->format('Y-m-d');
                break;
            case 'mes_atual':
                $dataInicio = Carbon::now()->startOfMonth()->format('Y-m-d');
                $dataFim = Carbon::today()->format('Y-m-d');
                break;
            case 'ultimo_trimestre':
                $dataInicio = Carbon::now()->subMonths(3)->startOfMonth()->format('Y-m-d');
                $dataFim = Carbon::today()->format('Y-m-d');
                break;
            case 'ultimo_ano':
                $dataInicio = Carbon::now()->subYear()->startOfYear()->format('Y-m-d');
                $dataFim = Carbon::now()->subYear()->endOfYear()->format('Y-m-d');
                break;
            case 'ano_atual':
                $dataInicio = Carbon::now()->startOfYear()->format('Y-m-d');
                $dataFim = Carbon::today()->format('Y-m-d');
                break;
            case 'personalizado':
                if (empty($dataInicio)) $dataInicio = Carbon::now()->startOfMonth()->format('Y-m-d');
                if (empty($dataFim)) $dataFim = Carbon::today()->format('Y-m-d');
                break;
            default:
                $dataInicio = Carbon::now()->startOfMonth()->format('Y-m-d');
                $dataFim = Carbon::today()->format('Y-m-d');
        }

        return [$dataInicio, $dataFim];
    }
}

