<?php

namespace App\Http\Controllers;

use App\Models\Carteira;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class UsuarioController extends Controller
{
    /**
     * Lista todos os usuários SAAS
     */
    public function index()
    {
        $user = Auth::user();
        
        // Verificar se é super admin
        if ($user->tipo != 1) {
            abort(403, 'Acesso negado');
        }
        
        // Estatísticas gerais
        $total_usuarios = Carteira::where('tipo', '!=', '1')->count();
        $total_aprovados = \Illuminate\Support\Facades\DB::table('payments')
            ->where('status', 'approved')
            ->count();
        $usuariosAtivos = Carteira::where('pagamentos', '2')->count();
        $total_com_login = Carteira::where('tipo', '!=', '1')
            ->whereNotNull('login')
            ->where('login', '!=', '')
            ->count();
        $total_com_assinatura = Carteira::where('tipo', '!=', '1')
            ->whereNotNull('assinatura')
            ->where('assinatura', '!=', '')
            ->count();

        // Novas estatísticas
        $total_com_whatsapp = 0;
        $total_com_pix = 0;
        $total_com_mercadopago = 0;
        $total_com_asaas = 0;
        $media_juros = 0;

        try {
            // Usuários com WhatsApp conectado
            if (\Illuminate\Support\Facades\DB::getSchemaBuilder()->hasTable('conexoes')) {
                $total_com_whatsapp = \Illuminate\Support\Facades\DB::table('conexoes as c')
                    ->join('carteiras as ca', 'c.id_usuario', '=', 'ca.id')
                    ->where('c.conn', 1)
                    ->where('ca.tipo', '!=', '1')
                    ->distinct('c.id_usuario')
                    ->count('c.id_usuario');
            }

            // Usuários com PIX
            $total_com_pix = Carteira::where('tipo', '!=', '1')
                ->whereNotNull('chave_pix')
                ->where('chave_pix', '!=', '')
                ->count();

            // Usuários com MercadoPago
            $total_com_mercadopago = Carteira::where('tipo', '!=', '1')
                ->whereNotNull('tokenmp')
                ->where('tokenmp', '!=', '')
                ->count();

            // Usuários com Asaas
            $total_com_asaas = Carteira::where('tipo', '!=', '1')
                ->whereNotNull('tokenasaas')
                ->where('tokenasaas', '!=', '')
                ->count();

            // Média de juros
            $media_juros = Carteira::where('tipo', '!=', '1')
                ->avg('juros_diarios') ?? 0;
        } catch (\Exception $e) {
            // Tabelas podem não existir, continuar
        }
        
        // Lista usuários que não são admin (tipo != 1)
        $usuarios = Carteira::where('tipo', '!=', '1')
            ->orderBy('nome', 'asc')
            ->get();

        // Buscar planos para o modal
        $planos = \App\Models\Plano::orderBy('nome')->get();

        return view('usuarios.index', compact(
            'usuarios', 'total_usuarios', 'total_aprovados', 'usuariosAtivos',
            'total_com_login', 'total_com_assinatura', 'total_com_whatsapp',
            'total_com_pix', 'total_com_mercadopago', 'total_com_asaas',
            'media_juros', 'planos'
        ));
    }

    /**
     * Exibe formulário de criar usuário
     */
    public function create()
    {
        return view('usuarios.create');
    }

    /**
     * Salva novo usuário
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'nome' => 'required|max:120',
            'celular' => 'required',
            'login' => 'required|email|unique:carteiras,login',
            'senha' => 'required|min:6',
            'assinatura' => 'required',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        Carteira::create([
            'idm' => $user->id,
            'nome' => $request->nome,
            'celular' => $request->celular,
            'login' => $request->login,
            'senha' => sha1($request->senha),
            'assinatura' => $request->assinatura,
            'tipo' => '2', // Revendedor
            'status' => '1', // Ativo
        ]);

        return redirect()->route('usuarios.index')->with('success', 'Usuário cadastrado com sucesso!');
    }

    /**
     * Exibe detalhes do usuário
     */
    public function show($id)
    {
        $user = Auth::user();
        
        $usuario = Carteira::where('id', $id)
            ->where('idm', $user->id)
            ->where('tipo', '!=', '1')
            ->firstOrFail();

        // Estatísticas do usuário
        $totalClientes = $usuario->clientes()->count();
        $totalCobrancas = $usuario->financeiro1()->count();
        $totalRecebido = $usuario->financeiro2()->where('status', '2')->sum('parcela');

        return view('usuarios.show', compact('usuario', 'totalClientes', 'totalCobrancas', 'totalRecebido'));
    }

    /**
     * Exibe formulário de edição
     */
    public function edit($id)
    {
        $user = Auth::user();
        
        $usuario = Carteira::where('id', $id)
            ->where('idm', $user->id)
            ->where('tipo', '!=', '1')
            ->firstOrFail();

        return view('usuarios.edit', compact('usuario'));
    }

    /**
     * Atualiza usuário
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();

        $usuario = Carteira::where('id', $id)
            ->where('idm', $user->id)
            ->where('tipo', '!=', '1')
            ->firstOrFail();

        $validator = Validator::make($request->all(), [
            'nome' => 'required|max:120',
            'celular' => 'required',
            'login' => 'required|email|unique:carteiras,login,' . $id,
            'assinatura' => 'required',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $data = [
            'nome' => $request->nome,
            'celular' => $request->celular,
            'login' => $request->login,
            'assinatura' => $request->assinatura,
        ];

        // Atualiza senha se fornecida
        if ($request->filled('senha')) {
            $data['senha'] = sha1($request->senha);
        }

        $usuario->update($data);

        return redirect()->route('usuarios.index')->with('success', 'Usuário atualizado com sucesso!');
    }

    /**
     * Deleta usuário
     */
    public function destroy($id)
    {
        $user = Auth::user();

        $usuario = Carteira::where('id', $id)
            ->where('idm', $user->id)
            ->where('tipo', '!=', '1')
            ->firstOrFail();

        // Verifica se tem dados associados
        if ($usuario->clientes()->count() > 0) {
            return back()->with('error', 'Não é possível excluir usuário com clientes associados.');
        }

        $usuario->delete();

        return redirect()->route('usuarios.index')->with('success', 'Usuário excluído com sucesso!');
    }
}
