<?php

namespace App\Http\Controllers;

use App\Models\Video;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

/**
 * Controller para gerenciar vídeos/tutoriais
 * Migrado do sistema legado: /legado/master/tutoriais.php
 */
class VideoController extends Controller
{
    /**
     * Lista todos os vídeos
     */
    public function index()
    {
        $user = Auth::user();
        
        $videos = Video::orderBy('created_at', 'desc')->get();

        return view('videos.index', compact('videos', 'user'));
    }

    /**
     * Salva novo vídeo (Ajax)
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        
        // Apenas admin pode adicionar vídeos
        if ($user->tipo != 1) {
            return response()->json([
                'success' => false, 
                'message' => 'Acesso negado. Apenas administradores podem adicionar vídeos.'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'required|max:255',
            'link' => 'required|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Dados inválidos.',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            Video::create([
                'title' => $request->title,
                'link' => $request->link,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Vídeo adicionado com sucesso!',
                'redirect' => route('videos.index', ['message' => 'Video+adicionado+com+sucesso'])
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao adicionar vídeo: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Formulário de edição
     */
    public function edit($id)
    {
        $user = Auth::user();
        
        // Apenas admin pode editar
        if ($user->tipo != 1) {
            abort(403, 'Acesso negado.');
        }

        $video = Video::findOrFail($id);

        return view('videos.edit', compact('video'));
    }

    /**
     * Atualiza vídeo
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        
        // Apenas admin pode editar
        if ($user->tipo != 1) {
            abort(403, 'Acesso negado.');
        }

        $video = Video::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'title' => 'required|max:255',
            'link' => 'required|max:255',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $video->update([
            'title' => $request->title,
            'link' => $request->link,
        ]);

        return redirect()->route('videos.index')
            ->with('success', 'Vídeo editado com sucesso!');
    }

    /**
     * Deleta vídeo
     */
    public function destroy($id)
    {
        $user = Auth::user();
        
        // Apenas admin pode deletar
        if ($user->tipo != 1) {
            return response()->json([
                'success' => false,
                'message' => 'Acesso negado.'
            ], 403);
        }

        try {
            $video = Video::findOrFail($id);
            $video->delete();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Vídeo deletado com sucesso!'
                ]);
            }

            return redirect()->route('videos.index')
                ->with('success', 'Vídeo deletado com sucesso!');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erro ao deletar vídeo.'
                ], 500);
            }

            return back()->with('error', 'Erro ao deletar vídeo.');
        }
    }
}
