<?php

namespace App\Http\Controllers;

use App\Models\Financeiro1;
use App\Models\Financeiro2;
use App\Models\FinanceiroGlobal;
use App\Models\Carteira;
use App\Models\Cliente;
use App\Models\Mensagem;
use App\Models\Conexao;
use App\Services\MercadoPagoService;
use App\Services\WhatsAppService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

/**
 * Controller para receber webhooks dos gateways de pagamento
 * Migrado do sistema legado: notificações de pagamento
 */
class WebhookController extends Controller
{
    /**
     * Webhook do Mercado Pago
     */
    public function mercadoPago(Request $request)
    {
        try {
            Log::info('Webhook Mercado Pago recebido', $request->all());

            // Suporta GET e POST (legado usa ambos)
            $data = $request->all();
            if ($request->isMethod('get')) {
                $data = $request->query();
            }

            // Extrair ID do pagamento
            $paymentId = $data['data']['id'] ?? $data['resource'] ?? $data['id'] ?? null;

            if (!$paymentId) {
                Log::warning('Webhook MP: ID não encontrado', ['data' => $data]);
                return response()->json(['status' => 'invalid'], 400);
            }

            // Busca pagamento em financeiro_global
            $finGlobal = FinanceiroGlobal::where('id_payment', $paymentId)->first();
            
            if (!$finGlobal) {
                Log::warning('Webhook MP: Pagamento não encontrado', ['payment_id' => $paymentId]);
                return response()->json(['status' => 'not_found'], 404);
            }

            // Verifica se já foi processado
            if ($finGlobal->status_payment === 'approved') {
                Log::info('Webhook MP: Pagamento já processado', ['payment_id' => $paymentId]);
                return response()->json(['status' => 'already_processed'], 200);
            }

            $carteira = Carteira::find($finGlobal->idm);
            if (!$carteira || !$carteira->tokenmp) {
                Log::error('Webhook MP: Token não configurado', ['idm' => $finGlobal->idm]);
                return response()->json(['status' => 'no_token'], 400);
            }

            // Consulta status na API
            $mpService = new MercadoPagoService($carteira->tokenmp);
            $status = $mpService->getPaymentStatus($paymentId);

            if (!$status || $status !== 'approved') {
                Log::warning('Webhook MP: Pagamento não aprovado', [
                    'payment_id' => $paymentId,
                    'status' => $status
                ]);
                return response()->json(['status' => 'not_approved'], 400);
            }

            // Atualiza status
            $this->updatePaymentStatus($finGlobal, $carteira, 'approved');

            return response()->json(['status' => 'success'], 200);

        } catch (\Exception $e) {
            Log::error('Erro ao processar webhook Mercado Pago', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'data' => $request->all()
            ]);

            return response()->json(['status' => 'error'], 500);
        }
    }

    /**
     * Webhook do Asaas
     */
    public function asaas(Request $request)
    {
        try {
            Log::info('Webhook Asaas recebido', $request->all());

            $event = $request->input('event');
            $payment = $request->input('payment');

            if (!$payment || !isset($payment['id'])) {
                return response()->json(['status' => 'invalid'], 400);
            }

            // Busca pagamento no financeiro_global
            $finGlobal = FinanceiroGlobal::where('id_payment', $payment['id'])->first();
            
            if (!$finGlobal) {
                Log::warning('Webhook Asaas: Pagamento não encontrado', ['payment_id' => $payment['id']]);
                return response()->json(['status' => 'not_found'], 404);
            }

            // Verifica se já foi processado
            if ($finGlobal->status_payment === 'approved') {
                Log::info('Webhook Asaas: Pagamento já processado', ['payment_id' => $payment['id']]);
                return response()->json(['status' => 'already_processed'], 200);
            }

            $carteira = Carteira::find($finGlobal->idm);
            if (!$carteira) {
                return response()->json(['status' => 'not_found'], 404);
            }

            // Atualiza status
            $status = $payment['status'] ?? '';
            $finGlobal->update(['status_payment' => $status]);

            // Se foi pago, dá baixa
            if (in_array($event, ['PAYMENT_RECEIVED', 'PAYMENT_CONFIRMED'])) {
                $this->updatePaymentStatus($finGlobal, $carteira, 'approved', $payment['paymentDate'] ?? null);
            }

            return response()->json(['status' => 'success'], 200);

        } catch (\Exception $e) {
            Log::error('Erro ao processar webhook Asaas', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'data' => $request->all()
            ]);

            return response()->json(['status' => 'error'], 500);
        }
    }

    /**
     * Webhook da Gerencianet
     */
    public function gerencianet(Request $request)
    {
        try {
            Log::info('Webhook Gerencianet recebido', $request->all());

            $pix = $request->input('pix');

            if (!$pix || !isset($pix[0]['txid'])) {
                return response()->json(['status' => 'invalid'], 400);
            }

            $txid = $pix[0]['txid'];

            // Busca pagamento
            $finGlobal = FinanceiroGlobal::where('id_payment', $txid)->first();
            
            if (!$finGlobal) {
                Log::warning('Webhook Gerencianet: Pagamento não encontrado', ['txid' => $txid]);
                return response()->json(['status' => 'not_found'], 404);
            }

            // Verifica se já foi processado
            if ($finGlobal->status_payment === 'CONCLUIDA') {
                Log::info('Webhook Gerencianet: Pagamento já processado', ['txid' => $txid]);
                return response()->json(['status' => 'already_processed'], 200);
            }

            $carteira = Carteira::find($finGlobal->idm);
            if (!$carteira) {
                return response()->json(['status' => 'not_found'], 404);
            }

            // Se foi pago
            if (isset($pix[0]['horario'])) {
                $this->updatePaymentStatus($finGlobal, $carteira, 'CONCLUIDA', $pix[0]['horario']);
            }

            return response()->json(['status' => 'success'], 200);

        } catch (\Exception $e) {
            Log::error('Erro ao processar webhook Gerencianet', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'data' => $request->all()
            ]);

            return response()->json(['status' => 'error'], 500);
        }
    }

    /**
     * Webhook do Paghiper
     */
    public function paghiper(Request $request)
    {
        try {
            Log::info('Webhook Paghiper recebido', $request->all());

            $transactionId = $request->input('transaction_id');
            $status = $request->input('status');

            if (!$transactionId) {
                return response()->json(['status' => 'invalid'], 400);
            }

            // Busca pagamento
            $finGlobal = FinanceiroGlobal::where('id_payment', $transactionId)->first();
            
            if (!$finGlobal) {
                Log::warning('Webhook Paghiper: Pagamento não encontrado', ['transaction_id' => $transactionId]);
                return response()->json(['status' => 'not_found'], 404);
            }

            // Verifica se já foi processado
            if ($finGlobal->status_payment === 'approved') {
                Log::info('Webhook Paghiper: Pagamento já processado', ['transaction_id' => $transactionId]);
                return response()->json(['status' => 'already_processed'], 200);
            }

            $carteira = Carteira::find($finGlobal->idm);
            if (!$carteira) {
                return response()->json(['status' => 'not_found'], 404);
            }

            // Atualiza status
            $finGlobal->update(['status_payment' => $status]);

            // Se foi pago
            if (in_array($status, ['paid', 'completed'])) {
                $this->updatePaymentStatus($finGlobal, $carteira, 'approved');
            }

            return response()->json(['status' => 'success'], 200);

        } catch (\Exception $e) {
            Log::error('Erro ao processar webhook Paghiper', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'data' => $request->all()
            ]);

            return response()->json(['status' => 'error'], 500);
        }
    }

    /**
     * Atualiza status do pagamento e envia notificação WhatsApp
     */
    private function updatePaymentStatus(FinanceiroGlobal $finGlobal, Carteira $carteira, string $status, ?string $paymentDate = null): void
    {
        DB::beginTransaction();
        try {
            // Atualiza financeiro_global
            $finGlobal->update(['status_payment' => $status]);

            // Busca parcela
            $parcela = Financeiro2::find($finGlobal->instancia);
            if (!$parcela) {
                throw new \Exception('Parcela não encontrada');
            }

            // Data de pagamento
            $dataPagamento = $paymentDate 
                ? Carbon::parse($paymentDate)->format('d/m/Y')
                : now()->format('d/m/Y');

            // Atualiza parcela
            $parcela->update([
                'status' => '2',
                'pagoem' => $dataPagamento
            ]);

            // Verifica se todas as parcelas foram pagas
            $chave = $parcela->chave;
            $todasPagas = Financeiro2::where('chave', $chave)
                ->where('status', '!=', '2')
                ->count() === 0;

            if ($todasPagas) {
                Financeiro1::where('chave', $chave)
                    ->where('status', '!=', '2')
                    ->update([
                        'status' => '2',
                        'pagoem' => $dataPagamento
                    ]);
            }

            DB::commit();

            // Envia notificação WhatsApp
            $this->sendWhatsAppNotification($parcela, $carteira, $finGlobal);

            Log::info('Pagamento processado com sucesso', [
                'parcela_id' => $parcela->id,
                'payment_id' => $finGlobal->id_payment,
                'status' => $status,
                'valor' => $parcela->parcela
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Erro ao atualizar status do pagamento', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Envia notificação WhatsApp após pagamento
     */
    private function sendWhatsAppNotification(Financeiro2 $parcela, Carteira $carteira, FinanceiroGlobal $finGlobal): void
    {
        try {
            // Busca cliente
            $cliente = Cliente::find($parcela->idc);
            if (!$cliente || !$cliente->celular) {
                Log::warning('WhatsApp: Cliente não encontrado ou sem telefone', [
                    'cliente_id' => $parcela->idc
                ]);
                return;
            }

            // Busca mensagem tipo 5 (recibo de pagamento)
            $mensagem = Mensagem::where('tipo', '5')
                ->where('idu', $carteira->Id)
                ->first();

            if (!$mensagem) {
                Log::warning('WhatsApp: Mensagem tipo 5 não encontrada', [
                    'idu' => $carteira->Id
                ]);
                return;
            }

            // Prepara dados para substituição
            $nameParts = explode(' ', $cliente->nome);
            $firstName = $nameParts[0] ?? '';
            $lastName = end($nameParts) ?? '';

            // Busca informações de produtos (se houver)
            $produtoInfo = 'Produto não especificado';
            if (!empty($parcela->produtoId)) {
                $ids = array_filter(array_map('trim', explode(',', $parcela->produtoId)));
                if (!empty($ids)) {
                    $produtos = DB::table('_pdv_produtos')
                        ->whereIn('id', $ids)
                        ->pluck('nome')
                        ->toArray();
                    $produtoInfo = !empty($produtos) ? implode(', ', $produtos) : 'Produto não encontrado';
                }
            }

            // Substitui variáveis na mensagem
            $texto = $mensagem->msg;
            $search = [
                '#NOME#', '#VENCIMENTO#', '#VALOR#', '#EMPRESA#', 
                '#CNPJ#', '#ENDERECO#', '#CONTATO#', '#DATAPAGAMENTO#', 
                '#TYPE#', '#PRODUTO#'
            ];
            $replace = [
                $firstName . ' ' . $lastName,
                $parcela->datapagamento ?? '',
                number_format($parcela->parcela, 2, ',', '.'),
                $carteira->nomecom ?? '',
                $carteira->cnpj ?? '',
                $carteira->enderecom ?? '',
                $cliente->celular ?? '',
                now()->format('d/m/Y'),
                strtoupper($finGlobal->type ?? 'PIX'),
                $produtoInfo
            ];
            $textoFinal = str_replace($search, $replace, $texto);
            $textoFinal = str_replace("\n", "\\n", $textoFinal);

            // Obtém instância padrão do WhatsApp
            $tokenapi = $this->obterApikeyPadrao($carteira->Id);
            if (!$tokenapi) {
                $tokenapi = $carteira->tokenapi;
            }

            if (!$tokenapi) {
                Log::warning('WhatsApp: Token não encontrado', ['idm' => $carteira->Id]);
                return;
            }

            // Obtém dados do servidor
            $dadosServidor = $this->getDadosServidor($tokenapi, $carteira->Id);
            if (!$dadosServidor) {
                Log::warning('WhatsApp: Dados do servidor não encontrados', ['idm' => $carteira->Id]);
                return;
            }

            // Envia mensagem via WhatsApp
            $whatsappService = new WhatsAppService(
                $dadosServidor['urlapi'],
                $dadosServidor['token'],
                $tokenapi
            );

            $success = $whatsappService->sendTextMessage($cliente->celular, $textoFinal);

            if ($success) {
                Log::info('WhatsApp: Notificação enviada com sucesso', [
                    'cliente_id' => $cliente->id,
                    'telefone' => $cliente->celular
                ]);
            } else {
                Log::error('WhatsApp: Erro ao enviar notificação', [
                    'cliente_id' => $cliente->id,
                    'telefone' => $cliente->celular
                ]);
            }

        } catch (\Exception $e) {
            Log::error('WhatsApp: Exceção ao enviar notificação', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Obtém apikey da instância padrão do WhatsApp
     */
    private function obterApikeyPadrao(int $idUsuario): ?string
    {
        try {
            // Busca instância padrão
            $instancia = DB::table('instancias_usuario')
                ->where('id_usuario', $idUsuario)
                ->where('ativo', 1)
                ->where('padrao', 1)
                ->whereNotNull('apikey')
                ->where('apikey', '!=', '')
                ->value('apikey');

            if ($instancia) {
                return $instancia;
            }

            // Busca qualquer instância ativa
            $instancia = DB::table('instancias_usuario')
                ->where('id_usuario', $idUsuario)
                ->where('ativo', 1)
                ->whereNotNull('apikey')
                ->where('apikey', '!=', '')
                ->orderBy('prioridade', 'asc')
                ->orderBy('id', 'asc')
                ->value('apikey');

            return $instancia;

        } catch (\Exception $e) {
            Log::error('Erro ao obter apikey padrão', [
                'id_usuario' => $idUsuario,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Obtém dados do servidor WhatsApp (URL e token)
     */
    private function getDadosServidor(string $tokenapi, int $idUsuario): ?array
    {
        try {
            // Busca instância para obter servidor_id
            $instancia = DB::table('instancias_usuario')
                ->where('apikey', $tokenapi)
                ->where('id_usuario', $idUsuario)
                ->where('ativo', 1)
                ->first();

            if ($instancia && $instancia->servidor_id) {
                $carteira = Carteira::find($idUsuario);
                if (!$carteira) {
                    return null;
                }

                // Determina qual servidor usar (1 ou 2)
                $servidorId = $instancia->servidor_id;
                
                if ($servidorId == '1') {
                    return [
                        'urlapi' => $carteira->whatsapp_api ?? '',
                        'token' => $carteira->whatsapp_token ?? ''
                    ];
                } elseif ($servidorId == '2') {
                    return [
                        'urlapi' => $carteira->whatsapp_api_2 ?? '',
                        'token' => $carteira->whatsapp_token_2 ?? ''
                    ];
                }
            }

            // Fallback: usa servidor padrão
            $carteira = Carteira::find($idUsuario);
            if ($carteira) {
                return [
                    'urlapi' => $carteira->whatsapp_api ?? '',
                    'token' => $carteira->whatsapp_token ?? ''
                ];
            }

            return null;

        } catch (\Exception $e) {
            Log::error('Erro ao obter dados do servidor', [
                'tokenapi' => substr($tokenapi, 0, 10) . '...',
                'id_usuario' => $idUsuario,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }
}
