<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Hash;

class Funcionario extends Model
{
    use HasFactory;

    protected $table = 'funcionarios';

    protected $fillable = [
        'id_saas',
        'nome',
        'celular',
        'login',
        'senha',
        'roles',
        'porcentagem_vendas',
        'status',
        'funcoes',
    ];

    protected $casts = [
        'porcentagem_vendas' => 'decimal:2',
        'status' => 'boolean',
    ];

    protected $hidden = [
        'senha',
    ];

    /**
     * Relacionamento com usuário SaaS (Carteira)
     */
    public function usuario(): BelongsTo
    {
        return $this->belongsTo(Carteira::class, 'id_saas', 'id');
    }

    /**
     * Retorna array de funções habilitadas
     */
    public function getFuncoesArrayAttribute(): array
    {
        return $this->funcoes ? explode(',', $this->funcoes) : [];
    }

    /**
     * Verifica se o funcionário tem uma função específica
     */
    public function temFuncao(string $funcao): bool
    {
        return in_array($funcao, $this->funcoes_array);
    }

    /**
     * Relacionamento com vendas
     */
    public function vendas(): HasMany
    {
        return $this->hasMany(PdvVenda::class, 'funcionario_id');
    }

    /**
     * Scope para funcionários ativos
     */
    public function scopeAtivos($query)
    {
        return $query->where('status', true);
    }

    /**
     * Verifica se o funcionário está ativo
     */
    public function isAtivo(): bool
    {
        return $this->status === true;
    }

    /**
     * Hash da senha ao salvar
     */
    public function setSenhaAttribute($value)
    {
        $this->attributes['senha'] = Hash::make($value);
    }

    /**
     * Verifica a senha
     */
    public function verificarSenha(string $senha): bool
    {
        return Hash::check($senha, $this->senha);
    }
}

