<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class PdvProduto extends Model
{
    use HasFactory;

    protected $table = '_pdv_produtos';

    protected $fillable = [
        'idm',
        'nome',
        'descricao',
        'preco_venda',
        'preco_custo',
        'margem_lucro',
        'estoque',
        'estoque_minimo',
        'alerta_estoque',
        'codigo_barras',
        'unidade',
        'fabricante',
        'peso',
        'ncm',
        'cfop',
        'fornecedor_id',
        'categoria_id',
        'imagem',
    ];

    protected $casts = [
        'preco_venda' => 'decimal:2',
        'preco_custo' => 'decimal:2',
        'margem_lucro' => 'decimal:2',
        'estoque' => 'integer',
        'estoque_minimo' => 'integer',
        'alerta_estoque' => 'boolean',
        'peso' => 'decimal:2',
    ];

    /**
     * Relacionamento com usuário
     */
    public function usuario(): BelongsTo
    {
        return $this->belongsTo(User::class, 'idm', 'id');
    }

    /**
     * Relacionamento com fornecedor
     */
    public function fornecedor(): BelongsTo
    {
        return $this->belongsTo(Fornecedor::class, 'fornecedor_id');
    }

    /**
     * Relacionamento com categoria
     */
    public function categoria(): BelongsTo
    {
        return $this->belongsTo(Categoria::class, 'categoria_id');
    }

    /**
     * Relacionamento com itens de venda
     */
    public function itens(): HasMany
    {
        return $this->hasMany(PdvItem::class, 'id_produto');
    }

    /**
     * Scope para produtos com estoque baixo
     */
    public function scopeEstoqueBaixo($query)
    {
        return $query->whereColumn('estoque', '<=', 'estoque_minimo')
            ->where('alerta_estoque', true);
    }

    /**
     * Scope para buscar por código de barras
     */
    public function scopePorCodigoBarras($query, string $codigo)
    {
        return $query->where('codigo_barras', $codigo);
    }

    /**
     * Verifica se o estoque está baixo
     */
    public function isEstoqueBaixo(): bool
    {
        return $this->estoque <= $this->estoque_minimo && $this->alerta_estoque;
    }

    /**
     * Calcula a margem de lucro automaticamente
     */
    public function calcularMargemLucro(): float
    {
        if ($this->preco_custo > 0) {
            return (($this->preco_venda - $this->preco_custo) / $this->preco_custo) * 100;
        }
        return 0;
    }

    /**
     * Adiciona estoque
     */
    public function adicionarEstoque(int $quantidade): void
    {
        $this->increment('estoque', $quantidade);
    }

    /**
     * Remove estoque
     */
    public function removerEstoque(int $quantidade): void
    {
        $this->decrement('estoque', $quantidade);
    }
}

