<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * Service para integração com Gerencianet (Efí Bank)
 * Migrado do sistema legado: /legado/gerencianet/gerencianet.php
 */
class GerencianetService
{
    private string $clientId;
    private string $clientSecret;
    private string $chavePix;
    private ?string $certificadoPem;
    private bool $sandbox;

    public function __construct(
        string $clientId,
        string $clientSecret,
        string $chavePix,
        ?string $certificadoPem = null,
        bool $sandbox = false
    ) {
        $this->clientId = $clientId;
        $this->clientSecret = $clientSecret;
        $this->chavePix = $chavePix;
        $this->certificadoPem = $certificadoPem;
        $this->sandbox = $sandbox;
    }

    /**
     * Obtém o token de autenticação
     */
    private function getAccessToken(): ?string
    {
        try {
            $baseUrl = $this->sandbox 
                ? 'https://api-pix-h.gerencianet.com.br'
                : 'https://api-pix.gerencianet.com.br';

            $credentials = base64_encode($this->clientId . ':' . $this->clientSecret);

            $response = Http::withHeaders([
                'Authorization' => 'Basic ' . $credentials,
                'Content-Type' => 'application/json',
            ])
            ->withOptions([
                'verify' => false, // Certificado SSL
            ])
            ->post($baseUrl . '/oauth/token', [
                'grant_type' => 'client_credentials'
            ]);

            if ($response->successful()) {
                $data = $response->json();
                return $data['access_token'] ?? null;
            }

            Log::error('Gerencianet: Erro ao obter token', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            return null;

        } catch (\Exception $e) {
            Log::error('Gerencianet: Exceção ao obter token', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Gera cobrança PIX
     */
    public function gerarPix(array $dados): ?array
    {
        try {
            $token = $this->getAccessToken();
            if (!$token) {
                return null;
            }

            $baseUrl = $this->sandbox 
                ? 'https://api-pix-h.gerencianet.com.br'
                : 'https://api-pix.gerencianet.com.br';

            $txid = $this->generateTxid();

            $body = [
                'calendario' => [
                    'expiracao' => 3600 // 1 hora
                ],
                'devedor' => [
                    'cpf' => $dados['cpf'] ?? '',
                    'nome' => $dados['nome'] ?? ''
                ],
                'valor' => [
                    'original' => number_format($dados['valor'], 2, '.', '')
                ],
                'chave' => $this->chavePix,
                'solicitacaoPagador' => $dados['descricao'] ?? 'Pagamento'
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
            ])
            ->withOptions([
                'verify' => false,
            ])
            ->put($baseUrl . '/v2/cob/' . $txid, $body);

            if ($response->successful()) {
                $cobranca = $response->json();
                
                // Gera QR Code
                $qrCode = $this->gerarQRCode($cobranca['loc']['id'], $token);
                
                return [
                    'txid' => $txid,
                    'id' => $cobranca['loc']['id'] ?? null,
                    'location' => $cobranca['location'] ?? null,
                    'status' => $cobranca['status'] ?? null,
                    'qrcode' => $qrCode['imagemQrcode'] ?? null,
                    'qrcode_text' => $qrCode['qrcode'] ?? null,
                ];
            }

            Log::error('Gerencianet: Erro ao gerar PIX', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            return null;

        } catch (\Exception $e) {
            Log::error('Gerencianet: Exceção ao gerar PIX', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Gera QR Code da cobrança
     */
    private function gerarQRCode(int $locId, string $token): ?array
    {
        try {
            $baseUrl = $this->sandbox 
                ? 'https://api-pix-h.gerencianet.com.br'
                : 'https://api-pix.gerencianet.com.br';

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])
            ->withOptions([
                'verify' => false,
            ])
            ->get($baseUrl . '/v2/loc/' . $locId . '/qrcode');

            if ($response->successful()) {
                return $response->json();
            }

            return null;

        } catch (\Exception $e) {
            Log::error('Gerencianet: Erro ao gerar QR Code', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Consulta status de cobrança PIX
     */
    public function consultarStatusPix(string $txid): ?array
    {
        try {
            $token = $this->getAccessToken();
            if (!$token) {
                return null;
            }

            $baseUrl = $this->sandbox 
                ? 'https://api-pix-h.gerencianet.com.br'
                : 'https://api-pix.gerencianet.com.br';

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])
            ->withOptions([
                'verify' => false,
            ])
            ->get($baseUrl . '/v2/cob/' . $txid);

            if ($response->successful()) {
                $data = $response->json();
                return [
                    'status' => $data['status'] ?? null,
                    'valor' => $data['valor']['original'] ?? null,
                    'pago' => ($data['status'] ?? '') === 'CONCLUIDA',
                ];
            }

            return null;

        } catch (\Exception $e) {
            Log::error('Gerencianet: Erro ao consultar status', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Gera TXID único
     */
    private function generateTxid(): string
    {
        return substr(md5(uniqid(rand(), true)), 0, 32);
    }
}

