<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * Service para integração com Paghiper
 * Migrado do sistema legado: /legado/paghiper/gerarpagamentos.php
 */
class PaghiperService
{
    private string $apiKey;
    private string $token;

    public function __construct(string $apiKey, string $token)
    {
        $this->apiKey = $apiKey;
        $this->token = $token;
    }

    /**
     * Gera boleto bancário
     */
    public function gerarBoleto(array $dados): ?array
    {
        try {
            $body = [
                'apiKey' => $this->apiKey,
                'order_id' => $dados['order_id'] ?? uniqid(),
                'payer_email' => $dados['email'] ?? '',
                'payer_name' => $dados['nome'] ?? '',
                'payer_cpf_cnpj' => $dados['cpf'] ?? '',
                'payer_phone' => $dados['telefone'] ?? '',
                'payer_street' => $dados['endereco'] ?? '',
                'payer_number' => $dados['numero'] ?? '',
                'payer_complement' => $dados['complemento'] ?? '',
                'payer_district' => $dados['bairro'] ?? '',
                'payer_city' => $dados['cidade'] ?? '',
                'payer_state' => $dados['uf'] ?? '',
                'payer_zip_code' => $dados['cep'] ?? '',
                'notification_url' => $dados['notification_url'] ?? '',
                'discount_cents' => '0',
                'shipping_price_cents' => '0',
                'shipping_methods' => 'Sem frete',
                'fixed_description' => true,
                'type_bank_slip' => 'boletoA4',
                'days_due_date' => $dados['dias_vencimento'] ?? 3,
                'items' => [
                    [
                        'description' => $dados['descricao'] ?? 'Pagamento',
                        'quantity' => '1',
                        'item_id' => '1',
                        'price_cents' => (int)($dados['valor'] * 100)
                    ]
                ]
            ];

            $response = Http::post('https://api.paghiper.com/transaction/create/', $body);

            if ($response->successful()) {
                $data = $response->json();
                
                if (isset($data['create_request']['result']) && $data['create_request']['result'] === 'success') {
                    return [
                        'transaction_id' => $data['create_request']['transaction_id'] ?? null,
                        'order_id' => $data['create_request']['order_id'] ?? null,
                        'status' => $data['create_request']['status'] ?? null,
                        'url_slip' => $data['create_request']['bank_slip']['url_slip'] ?? null,
                        'url_slip_pdf' => $data['create_request']['bank_slip']['url_slip_pdf'] ?? null,
                        'digitable_line' => $data['create_request']['bank_slip']['digitable_line'] ?? null,
                        'bar_code_number_to_image' => $data['create_request']['bank_slip']['bar_code_number_to_image'] ?? null,
                        'due_date' => $data['create_request']['due_date'] ?? null,
                        'num_carteira' => $data['create_request']['bank_slip']['num_carteira'] ?? null,
                        'type_bank_slip' => $data['create_request']['bank_slip']['type_bank_slip'] ?? null,
                    ];
                }

                Log::error('Paghiper: Erro na resposta', [
                    'response' => $data
                ]);
                return null;
            }

            Log::error('Paghiper: Erro ao gerar boleto', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            return null;

        } catch (\Exception $e) {
            Log::error('Paghiper: Exceção ao gerar boleto', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Consulta status de boleto
     */
    public function consultarStatus(string $transactionId): ?array
    {
        try {
            $body = [
                'token' => $this->token,
                'apiKey' => $this->apiKey,
                'transaction_id' => $transactionId,
                'order_id' => ''
            ];

            $response = Http::post('https://api.paghiper.com/transaction/status/', $body);

            if ($response->successful()) {
                $data = $response->json();
                
                if (isset($data['status_request']['result']) && $data['status_request']['result'] === 'success') {
                    return [
                        'status' => $data['status_request']['status'] ?? null,
                        'pago' => ($data['status_request']['status'] ?? '') === 'paid',
                        'valor' => isset($data['status_request']['value_cents']) 
                            ? $data['status_request']['value_cents'] / 100 
                            : null,
                    ];
                }
            }

            return null;

        } catch (\Exception $e) {
            Log::error('Paghiper: Erro ao consultar status', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Cancela boleto
     */
    public function cancelarBoleto(string $transactionId): bool
    {
        try {
            $body = [
                'token' => $this->token,
                'apiKey' => $this->apiKey,
                'transaction_id' => $transactionId,
                'status' => 'canceled'
            ];

            $response = Http::post('https://api.paghiper.com/transaction/cancel/', $body);

            if ($response->successful()) {
                $data = $response->json();
                return isset($data['cancellation_request']['result']) 
                    && $data['cancellation_request']['result'] === 'success';
            }

            return false;

        } catch (\Exception $e) {
            Log::error('Paghiper: Erro ao cancelar boleto', [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
}

