<?php

namespace App\Services;

use App\Models\Carteira;
use Illuminate\Support\Facades\Log;

class PaymentService
{
    protected $carteira;

    public function __construct(Carteira $carteira)
    {
        $this->carteira = $carteira;
    }

    /**
     * Gera pagamento baseado no gateway configurado
     */
    public function gerarPagamento(array $dados, string $type = 'pix'): ?array
    {
        $gateway = $this->carteira->pagamentos ?? $this->carteira->gatewayPayment ?? null;

        switch ($gateway) {
            case '1': // Mercado Pago
                return $this->gerarPagamentoMercadoPago($dados, $type);
            
            case '2': // PIX Manual
                return $this->gerarPagamentoPixManual($dados);
            
            case '3': // PagHiper
                return $this->gerarPagamentoPagHiper($dados);
            
            case '4': // Gerencianet/Efí PIX
                return $this->gerarPagamentoGerencianet($dados);
            
            case '5': // Asaas
                return $this->gerarPagamentoAsaas($dados, $type);
            
            default:
                throw new \Exception('Gateway de pagamento não configurado');
        }
    }

    /**
     * Mercado Pago - Gerar PIX ou Boleto
     */
    private function gerarPagamentoMercadoPago(array $dados, string $type = 'pix'): ?array
    {
        $accessToken = $this->carteira->tokenmp;

        if (!$accessToken) {
            throw new \Exception('Token Mercado Pago não configurado');
        }

        try {
            $notificationUrl = url('/webhook/mercadopago');
            $mpService = new MercadoPagoService($accessToken, $notificationUrl);

            $options = [
                'item_valor' => $dados['valor'] ?? 0,
                'vencimento' => $dados['vencimento'] ?? null,
                'name' => $dados['name'] ?? $dados['nome'] ?? '',
                'cpf' => $dados['cpf'] ?? '',
                'email' => $dados['email'] ?? '',
                'item_description' => $dados['descricao'] ?? $dados['item_description'] ?? 'Pagamento',
                'cliente_db' => $dados['cliente_db'] ?? null,
            ];

            $result = $mpService->createPayment($options, $type);

            if ($result && !isset($result['error'])) {
                return [
                    'success' => true,
                    'type' => $type,
                    'id_payment' => $result['id_payment'] ?? '',
                    'status_payment' => $result['status_payment'] ?? 'pending',
                    'imagemQrcode' => $result['imagemQrcode'] ?? '',
                    'copiacola' => $result['copiacola'] ?? '',
                    'codigobarra' => $result['codigobarra'] ?? '',
                    'valorparcela' => $result['valorparcela'] ?? $dados['valor'] ?? 0,
                    'createdDate' => $result['createdDate'] ?? now()->format('Y-m-d H:i:s'),
                ];
            }

            throw new \Exception('Erro ao gerar pagamento: ' . ($result['message'] ?? 'Erro desconhecido'));

        } catch (\Exception $e) {
            Log::error('Erro Mercado Pago: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * PIX Manual - Gerar pagamento
     */
    private function gerarPagamentoPixManual(array $dados): ?array
    {
        $chavePix = $this->carteira->pix_manual_key ?? $this->carteira->chave_pix ?? '';

        if (empty($chavePix)) {
            throw new \Exception('Chave PIX Manual não configurada');
        }

        try {
            $pixService = new PixManualService($chavePix);

            $options = [
                'item_valor' => $dados['valor'] ?? 0,
            ];

            $result = $pixService->generatePayment($options);

            if ($result && !isset($result['error'])) {
                return [
                    'success' => true,
                    'type' => 'pix',
                    'id_payment' => $result['id_payment'] ?? '',
                    'status_payment' => $result['status_payment'] ?? 'pending',
                    'copiacola' => $result['copiacola'] ?? '',
                    'imagemQrcode' => $result['imagemQrcode'] ?? '',
                    'codigobarra' => '',
                    'valorparcela' => $result['valorparcela'] ?? $dados['valor'] ?? 0,
                    'createdDate' => $result['createdDate'] ?? now()->format('Y-m-d H:i:s'),
                ];
            }

            throw new \Exception('Erro ao gerar PIX Manual: ' . ($result['message'] ?? 'Erro desconhecido'));

        } catch (\Exception $e) {
            Log::error('Erro PIX Manual: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Gerencianet/Efí - Gerar PIX
     */
    private function gerarPagamentoGerencianet(array $dados): ?array
    {
        $clientId = $this->carteira->gerecianet_client;
        $clientSecret = $this->carteira->gerecianet_secret;
        $chavePix = $this->carteira->chave_pix;
        $certificado = $this->carteira->certificado_pem;

        if (!$clientId || !$clientSecret || !$chavePix) {
            throw new \Exception('Credenciais Gerencianet não configuradas');
        }

        try {
            $gerencianetService = new GerencianetService(
                $clientId,
                $clientSecret,
                $chavePix,
                $certificado,
                false // sandbox
            );

            $dadosPix = [
                'cpf' => $dados['cpf'] ?? '',
                'nome' => $dados['name'] ?? $dados['nome'] ?? '',
                'valor' => $dados['valor'] ?? 0,
                'descricao' => $dados['descricao'] ?? 'Pagamento',
            ];

            $result = $gerencianetService->gerarPix($dadosPix);

            if ($result) {
                return [
                    'success' => true,
                    'type' => 'pix',
                    'id_payment' => $result['txid'] ?? '',
                    'status_payment' => $result['status'] ?? 'pending',
                    'imagemQrcode' => $result['qrcode'] ?? '',
                    'copiacola' => $result['qrcode_text'] ?? '',
                    'codigobarra' => '',
                    'valorparcela' => $dados['valor'] ?? 0,
                    'createdDate' => now()->format('Y-m-d H:i:s'),
                ];
            }

            throw new \Exception('Erro ao gerar PIX Gerencianet');

        } catch (\Exception $e) {
            Log::error('Erro Gerencianet: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * PagHiper - Gerar Boleto
     */
    private function gerarPagamentoPagHiper(array $dados): ?array
    {
        $apiKey = $this->carteira->key_paghiper;
        $token = $this->carteira->token_paghiper;

        if (!$apiKey || !$token) {
            throw new \Exception('Credenciais PagHiper não configuradas');
        }

        try {
            $paghiperService = new PaghiperService($apiKey, $token);

            $dadosBoleto = [
                'order_id' => $dados['order_id'] ?? uniqid(),
                'email' => $dados['email'] ?? '',
                'nome' => $dados['name'] ?? $dados['nome'] ?? '',
                'cpf' => $dados['cpf'] ?? '',
                'telefone' => $dados['telefone'] ?? '',
                'endereco' => $dados['endereco'] ?? '',
                'numero' => $dados['numero'] ?? '',
                'complemento' => $dados['complemento'] ?? '',
                'bairro' => $dados['bairro'] ?? '',
                'cidade' => $dados['cidade'] ?? '',
                'uf' => $dados['uf'] ?? '',
                'cep' => $dados['cep'] ?? '',
                'valor' => $dados['valor'] ?? 0,
                'descricao' => $dados['descricao'] ?? 'Pagamento',
                'dias_vencimento' => $dados['dias_vencimento'] ?? 3,
                'notification_url' => url('/webhook/paghiper'),
            ];

            $result = $paghiperService->gerarBoleto($dadosBoleto);

            if ($result) {
                return [
                    'success' => true,
                    'type' => 'boleto',
                    'id_payment' => $result['transaction_id'] ?? '',
                    'status_payment' => $result['status'] ?? 'pending',
                    'codigobarra' => $result['digitable_line'] ?? '',
                    'url_boleto' => $result['url_slip_pdf'] ?? '',
                    'imagemQrcode' => '',
                    'copiacola' => '',
                    'valorparcela' => $dados['valor'] ?? 0,
                    'createdDate' => now()->format('Y-m-d H:i:s'),
                ];
            }

            throw new \Exception('Erro ao gerar boleto PagHiper');

        } catch (\Exception $e) {
            Log::error('Erro PagHiper: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Asaas - Gerar PIX ou Boleto
     */
    private function gerarPagamentoAsaas(array $dados, string $type = 'pix'): ?array
    {
        $apiKey = $this->carteira->tokenasaas;

        if (!$apiKey) {
            throw new \Exception('Token Asaas não configurado');
        }

        try {
            $asaasService = new AsaasService($apiKey, false);

            // Verifica se cliente existe, se não cria
            $customerId = $dados['customer_id'] ?? null;
            if (!$customerId && isset($dados['cpf']) && isset($dados['nome'])) {
                $customerData = [
                    'name' => $dados['name'] ?? $dados['nome'] ?? '',
                    'cpfCnpj' => str_replace(['.', '-', '/'], '', $dados['cpf'] ?? ''),
                    'email' => $dados['email'] ?? null,
                    'phone' => $dados['telefone'] ?? null,
                    'externalReference' => $dados['external_reference'] ?? null,
                ];
                $customerId = $asaasService->createClient($customerData);
            }

            if (!$customerId) {
                throw new \Exception('Erro ao criar/buscar cliente no Asaas');
            }

            // Formata data de vencimento
            $dueDate = $dados['vencimento'] ?? null;
            if ($dueDate) {
                try {
                    $date = \Carbon\Carbon::createFromFormat('d/m/Y', $dueDate);
                    $dueDate = $date->format('Y-m-d');
                } catch (\Exception $e) {
                    $dueDate = now()->addDays(3)->format('Y-m-d');
                }
            } else {
                $dueDate = now()->addDays(3)->format('Y-m-d');
            }

            // Cria pagamento
            $paymentData = [
                'customer' => $customerId,
                'billingType' => strtoupper($type) === 'PIX' ? 'PIX' : 'BOLETO',
                'value' => $dados['valor'] ?? 0,
                'dueDate' => $dueDate,
                'description' => $dados['descricao'] ?? 'Pagamento',
                'externalReference' => $dados['external_reference'] ?? null,
            ];

            $result = $asaasService->createPayment($paymentData);

            if (!$result || !isset($result['id'])) {
                throw new \Exception('Erro ao criar pagamento no Asaas');
            }

            $paymentId = $result['id'];
            $return = [
                'success' => true,
                'type' => strtolower($result['billingType'] ?? $type),
                'id_payment' => $paymentId,
                'status_payment' => $result['status'] ?? 'pending',
                'valorparcela' => $dados['valor'] ?? 0,
                'createdDate' => now()->format('Y-m-d H:i:s'),
            ];

            // Se for PIX, busca QR Code
            if (strtoupper($type) === 'PIX' || ($result['billingType'] ?? '') === 'PIX') {
                $qrcode = $asaasService->generatePixQrCode($paymentId);
                if ($qrcode) {
                    $return['imagemQrcode'] = $qrcode['encodedImage'] ?? '';
                    $return['copiacola'] = $qrcode['payload'] ?? '';
                }
                $return['codigobarra'] = '';
            } 
            // Se for Boleto
            else {
                $return['url_boleto'] = $result['bankSlipUrl'] ?? '';
                $return['digitable_line'] = $result['identificationField'] ?? '';
                $return['imagemQrcode'] = '';
                $return['copiacola'] = '';
            }

            return $return;

        } catch (\Exception $e) {
            Log::error('Erro Asaas: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Verifica status de um pagamento
     */
    public function verificarStatus(string $paymentId, string $gateway): ?array
    {
        switch ($gateway) {
            case '1': // Mercado Pago
                return $this->verificarStatusMercadoPago($paymentId);
            
            case '4': // Gerencianet
                return $this->verificarStatusGerencianet($paymentId);
            
            case '5': // Asaas
                return $this->verificarStatusAsaas($paymentId);
            
            default:
                throw new \Exception('Verificação não implementada para este gateway');
        }
    }

    private function verificarStatusMercadoPago(string $paymentId): ?array
    {
        $accessToken = $this->carteira->tokenmp;
        if (!$accessToken) {
            return null;
        }

        $mpService = new MercadoPagoService($accessToken);
        $payment = $mpService->getPayment($paymentId);

        return $payment ? [
            'status' => $payment['status'],
            'pago' => in_array($payment['status'], ['approved', 'in_process']),
            'valor' => $payment['transaction_amount'] ?? null,
        ] : null;
    }

    private function verificarStatusGerencianet(string $txid): ?array
    {
        $clientId = $this->carteira->gerecianet_client;
        $clientSecret = $this->carteira->gerecianet_secret;
        $chavePix = $this->carteira->chave_pix;

        if (!$clientId || !$clientSecret || !$chavePix) {
            return null;
        }

        $gerencianetService = new GerencianetService($clientId, $clientSecret, $chavePix);
        return $gerencianetService->consultarStatusPix($txid);
    }

    private function verificarStatusAsaas(string $paymentId): ?array
    {
        $apiKey = $this->carteira->tokenasaas;
        if (!$apiKey) {
            return null;
        }

        $asaasService = new AsaasService($apiKey);
        $status = $asaasService->getPaymentStatus($paymentId);

        return $status ? [
            'status' => $status['status'],
            'pago' => $status['pago'] ?? false,
            'valor' => $status['value'] ?? null,
        ] : null;
    }
}

