<?php

namespace App\Services;

use App\Models\MessageQueue;
use App\Models\Conexao;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * Service para integração com WhatsApp
 * Migrado do sistema legado: /legado/master/classes/gera_qr.php e functions.php
 */
class WhatsAppService
{
    private string $apiUrl;
    private string $apiKey;
    private string $instance;

    public function __construct(string $apiUrl, string $apiKey, string $instance)
    {
        $this->apiUrl = rtrim($apiUrl, '/');
        $this->apiKey = $apiKey;
        $this->instance = 'AbC123' . $instance; // Padrão do legado
    }

    /**
     * Gera QR Code para conectar WhatsApp
     */
    public function generateQRCode(): ?array
    {
        try {
            $response = Http::withHeaders([
                'apikey' => $this->apiKey
            ])->get($this->apiUrl . '/instance/connect/' . $this->instance);

            if ($response->successful()) {
                $data = $response->json();
                return [
                    'qrcode' => $data['qrcode']['base64'] ?? null,
                    'code' => $data['qrcode']['code'] ?? null,
                ];
            }

            Log::error('WhatsApp: Erro ao gerar QR Code', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            return null;

        } catch (\Exception $e) {
            Log::error('WhatsApp: Exceção ao gerar QR Code', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Verifica status da conexão
     */
    public function getConnectionStatus(): ?array
    {
        try {
            $response = Http::withHeaders([
                'apikey' => $this->apiKey
            ])->get($this->apiUrl . '/instance/connectionState/' . $this->instance);

            if ($response->successful()) {
                $data = $response->json();
                return [
                    'state' => $data['instance']['state'] ?? 'close',
                    'connected' => ($data['instance']['state'] ?? '') === 'open',
                ];
            }

            return ['state' => 'close', 'connected' => false];

        } catch (\Exception $e) {
            Log::error('WhatsApp: Erro ao verificar status', [
                'error' => $e->getMessage()
            ]);
            return ['state' => 'close', 'connected' => false];
        }
    }

    /**
     * Desconecta WhatsApp
     */
    public function disconnect(): bool
    {
        try {
            $response = Http::withHeaders([
                'apikey' => $this->apiKey
            ])->delete($this->apiUrl . '/instance/logout/' . $this->instance);

            return $response->successful();

        } catch (\Exception $e) {
            Log::error('WhatsApp: Erro ao desconectar', [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Envia mensagem de texto
     */
    public function sendTextMessage(string $phone, string $message): bool
    {
        try {
            // Formata telefone (remove caracteres não numéricos e adiciona 55)
            $phone = '55' . preg_replace('/[^0-9]/', '', $phone);

            $body = [
                'number' => $phone,
                'options' => [
                    'delay' => 1200,
                    'presence' => 'composing',
                    'linkPreview' => false
                ],
                'textMessage' => [
                    'text' => $message
                ]
            ];

            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'apikey' => $this->apiKey
            ])->post($this->apiUrl . '/message/sendText/' . $this->instance, $body);

            if ($response->successful()) {
                Log::info('WhatsApp: Mensagem enviada', [
                    'phone' => $phone,
                    'message_length' => strlen($message)
                ]);
                return true;
            }

            Log::error('WhatsApp: Erro ao enviar mensagem', [
                'phone' => $phone,
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            return false;

        } catch (\Exception $e) {
            Log::error('WhatsApp: Exceção ao enviar mensagem', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Envia mensagem com mídia (imagem)
     */
    public function sendMediaMessage(string $phone, string $caption, string $base64Image): bool
    {
        try {
            // Formata telefone
            $phone = '55' . preg_replace('/[^0-9]/', '', $phone);

            $body = [
                'number' => $phone,
                'options' => [
                    'delay' => 1200,
                    'presence' => 'composing'
                ],
                'mediaMessage' => [
                    'mediatype' => 'image',
                    'caption' => $caption,
                    'media' => $base64Image
                ]
            ];

            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'apikey' => $this->apiKey
            ])->post($this->apiUrl . '/message/sendMedia/' . $this->instance, $body);

            if ($response->successful()) {
                Log::info('WhatsApp: Mídia enviada', [
                    'phone' => $phone,
                    'caption' => $caption
                ]);
                return true;
            }

            Log::error('WhatsApp: Erro ao enviar mídia', [
                'phone' => $phone,
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            return false;

        } catch (\Exception $e) {
            Log::error('WhatsApp: Exceção ao enviar mídia', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Adiciona mensagem à fila
     */
    public function queueMessage(string $type, string $phone, string $message, ?string $media = null): void
    {
        MessageQueue::create([
            'type' => $type,
            'phone' => $phone,
            'message' => $message,
            'media' => $media,
            'status' => 'pending'
        ]);
    }

    /**
     * Processa fila de mensagens
     */
    public function processQueue(int $limit = 10): array
    {
        $messages = MessageQueue::where('status', 'pending')
            ->orderBy('id')
            ->limit($limit)
            ->get();

        $results = [
            'processed' => 0,
            'success' => 0,
            'failed' => 0,
        ];

        foreach ($messages as $message) {
            $results['processed']++;

            $success = false;

            if ($message->type === 'text') {
                $success = $this->sendTextMessage($message->phone, $message->message);
            } elseif ($message->type === 'media') {
                $success = $this->sendMediaMessage($message->phone, $message->message, $message->media);
            }

            if ($success) {
                $message->update(['status' => 'sent']);
                $results['success']++;
            } else {
                $message->update(['status' => 'failed']);
                $results['failed']++;
            }

            // Delay entre mensagens para evitar bloqueio
            sleep(2);
        }

        return $results;
    }

    /**
     * Substitui variáveis dinâmicas na mensagem
     */
    public function replaceVariables(string $message, array $data): string
    {
        $search = [
            '#NOME#',
            '#VENCIMENTO#',
            '#VALOR#',
            '#LINK#',
            '#EMPRESA#',
            '#CNPJ#',
            '#ENDERECO#',
            '#CONTATO#',
        ];

        $replace = [
            $data['nome'] ?? '',
            $data['vencimento'] ?? '',
            $data['valor'] ?? '',
            $data['link'] ?? '',
            $data['empresa'] ?? '',
            $data['cnpj'] ?? '',
            $data['endereco'] ?? '',
            $data['contato'] ?? '',
        ];

        return str_replace($search, $replace, $message);
    }

    /**
     * Limpa fila de mensagens antigas
     */
    public function cleanOldMessages(int $days = 30): int
    {
        return MessageQueue::where('created_at', '<', now()->subDays($days))
            ->whereIn('status', ['sent', 'failed'])
            ->delete();
    }
}
