@extends('layouts.app')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h4>Nova Venda</h4>
                    <a href="{{ route('pdv.index') }}" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> Voltar ao PDV
                    </a>
                </div>
                <div class="card-body">
                    @if(session('error'))
                        <div class="alert alert-danger">{{ session('error') }}</div>
                    @endif

                    <form id="form-venda" action="{{ route('pdv.vendas.store') }}" method="POST">
                        @csrf
                        <input type="hidden" name="caixa_id" value="{{ $caixa->id }}">

                        <!-- Cliente e Forma de Pagamento -->
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="busca-cliente" class="form-label">
                                    Cliente <span class="text-danger">*</span>
                                </label>
                                <div class="input-group">
                                    <span class="input-group-text"><i class="fas fa-user"></i></span>
                                    <input type="text" class="form-control" id="busca-cliente" 
                                        placeholder="Nome, CPF ou CNPJ" required>
                                    <input type="hidden" name="cliente_id" id="cliente_id">
                                </div>
                                <div id="clientes-lista" class="list-group position-absolute w-100" 
                                    style="z-index: 1050; display: none; max-height: 200px; overflow-y: auto; border: 1px solid #dee2e6; background: white;">
                                </div>
                                <small id="cliente-selecionado-nome" class="text-success"></small>
                            </div>

                            <div class="col-md-6">
                                <label for="forma_pagamento" class="form-label">
                                    Forma de Pagamento <span class="text-danger">*</span>
                                </label>
                                <select class="form-select" id="forma_pagamento" name="forma_pagamento" required>
                                    <option value="">Selecione...</option>
                                    @foreach($formasPagamento as $fp)
                                        <option value="{{ $fp->nome }}">{{ $fp->nome }}</option>
                                    @endforeach
                                </select>
                            </div>
                        </div>

                        <!-- Opções -->
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" id="lancar_conta" name="lancar_conta" value="1">
                                    <label class="form-check-label" for="lancar_conta">
                                        Lançar cobrança no Contas a Receber?
                                    </label>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" id="enviar_comprovante" name="enviar_comprovante" value="1" checked>
                                    <label class="form-check-label" for="enviar_comprovante">
                                        Enviar comprovante via WhatsApp?
                                    </label>
                                </div>
                            </div>
                        </div>

                        <!-- Campos de Parcelamento (ocultos por padrão) -->
                        <div id="campos-parcelamento" class="row mb-3" style="display: none;">
                            <div class="col-md-4">
                                <label for="parcelas" class="form-label">Quantidade de Parcelas</label>
                                <input type="number" class="form-control" id="parcelas" name="parcelas" min="1" max="12" value="1">
                            </div>
                            <div class="col-md-4">
                                <label for="data_vencimento" class="form-label">Data de Vencimento (Primeira Parcela)</label>
                                <input type="date" class="form-control" id="data_vencimento" name="data_vencimento" value="{{ date('Y-m-d', strtotime('+1 month')) }}">
                            </div>
                        </div>

                        <!-- Buscar Produtos -->
                        <div class="row mb-3">
                            <div class="col-md-12">
                                <h5 class="mb-3">Buscar Produtos</h5>
                                <div class="input-group mb-3">
                                    <span class="input-group-text"><i class="fas fa-search"></i></span>
                                    <input type="text" class="form-control" id="busca-produto" 
                                        placeholder="Digite o nome ou código de barras do produto...">
                                </div>
                                <div id="resultado-produtos" class="list-group position-absolute" 
                                    style="z-index: 1050; display: none; max-height: 300px; overflow-y: auto; width: 100%; max-width: 600px; border: 1px solid #dee2e6; background: white;">
                                </div>
                            </div>
                        </div>

                        <!-- Carrinho -->
                        <div class="row">
                            <div class="col-12">
                                <h5 class="mb-3"><i class="fas fa-shopping-cart"></i> Carrinho</h5>
                                <div class="table-responsive">
                                    <table class="table table-striped">
                                        <thead>
                                            <tr>
                                                <th>Produto</th>
                                                <th class="text-center" style="width: 100px;">Qtd</th>
                                                <th class="text-end" style="width: 120px;">Preço Unit.</th>
                                                <th class="text-end" style="width: 120px;">Subtotal</th>
                                                <th class="text-center" style="width: 80px;">Ações</th>
                                            </tr>
                                        </thead>
                                        <tbody id="carrinho-tbody">
                                            <tr id="carrinho-vazio">
                                                <td colspan="5" class="text-center text-muted py-4">
                                                    <i class="fas fa-shopping-cart fa-2x mb-2"></i><br>
                                                    Carrinho vazio
                                                </td>
                                            </tr>
                                        </tbody>
                                    </table>
                                </div>

                                <!-- Resumo -->
                                <div class="row mt-3">
                                    <div class="col-md-6 offset-md-6">
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>Subtotal:</span>
                                            <span id="subtotal">R$ 0,00</span>
                                        </div>
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>Desconto:</span>
                                            <input type="number" class="form-control form-control-sm" id="desconto-input" 
                                                name="desconto" step="0.01" min="0" value="0" style="width: 120px; display: inline-block;">
                                        </div>
                                        <hr>
                                        <div class="d-flex justify-content-between mb-3">
                                            <strong>Total:</strong>
                                            <strong id="total" style="font-size: 1.5rem; color: #198754;">R$ 0,00</strong>
                                        </div>
                                        <input type="hidden" name="total" id="total-hidden" value="0">
                                        <button type="submit" class="btn btn-success btn-lg w-100" id="btn-finalizar" disabled>
                                            <i class="fas fa-check"></i> Finalizar Venda
                                        </button>
                                        <small class="text-muted d-block text-center mt-2">
                                            <i class="fas fa-info-circle"></i> Cliente, produtos e forma de pagamento são obrigatórios
                                        </small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
let carrinho = [];
let clienteSelecionado = null;

$(document).ready(function() {
    // Toggle campos de parcelamento
    $('#lancar_conta').on('change', function() {
        if ($(this).is(':checked')) {
            $('#campos-parcelamento').show();
        } else {
            $('#campos-parcelamento').hide();
        }
    });

    // Buscar clientes
    let timeoutCliente;
    $('#busca-cliente').on('input', function() {
        clearTimeout(timeoutCliente);
        const termo = $(this).val();
        
        if (termo.length < 2) {
            $('#clientes-lista').hide();
            return;
        }

        timeoutCliente = setTimeout(function() {
            buscarClientes(termo);
        }, 300);
    });

    // Buscar produtos
    let timeoutProduto;
    $('#busca-produto').on('input', function() {
        clearTimeout(timeoutProduto);
        const termo = $(this).val();
        
        if (termo.length < 2) {
            $('#resultado-produtos').hide();
            return;
        }

        timeoutProduto = setTimeout(function() {
            buscarProdutos(termo);
        }, 300);
    });

    // Atualizar total quando desconto mudar
    $('#desconto-input').on('input', calcularTotal);
});

function buscarClientes(termo) {
    $.ajax({
        url: '{{ route("pdv.vendas.buscar-clientes") }}',
        method: 'GET',
        data: { termo: termo },
        success: function(clientes) {
            let html = '';
            if (clientes.length > 0) {
                clientes.forEach(function(cliente) {
                    html += `<a href="#" class="list-group-item list-group-item-action" onclick="selecionarCliente(${cliente.id}, '${cliente.nome}'); return false;">
                        <strong>${cliente.nome}</strong><br>
                        <small>${cliente.cpf || ''} ${cliente.email || ''}</small>
                    </a>`;
                });
            } else {
                html = '<div class="list-group-item">Nenhum cliente encontrado</div>';
            }
            $('#clientes-lista').html(html).show();
        }
    });
}

function selecionarCliente(id, nome) {
    clienteSelecionado = { id: id, nome: nome };
    $('#cliente_id').val(id);
    $('#busca-cliente').val(nome);
    $('#cliente-selecionado-nome').text('Cliente selecionado: ' + nome);
    $('#clientes-lista').hide();
    verificarFinalizacao();
}

function buscarProdutos(termo) {
    $.ajax({
        url: '{{ route("pdv.vendas.buscar-produtos") }}',
        method: 'GET',
        data: { termo: termo },
        success: function(produtos) {
            let html = '';
            if (produtos.length > 0) {
                produtos.forEach(function(produto) {
                    html += `<a href="#" class="list-group-item list-group-item-action" onclick="adicionarProduto(${produto.id}, '${produto.nome}', ${produto.preco_venda}, ${produto.estoque}); return false;">
                        <div class="d-flex align-items-center">
                            ${produto.imagem ? `<img src="${produto.imagem}" style="width: 50px; height: 50px; object-fit: cover; margin-right: 10px;">` : ''}
                            <div>
                                <strong>${produto.nome}</strong><br>
                                <small>R$ ${parseFloat(produto.preco_venda).toFixed(2).replace('.', ',')} | Estoque: ${produto.estoque}</small>
                            </div>
                        </div>
                    </a>`;
                });
            } else {
                html = '<div class="list-group-item">Nenhum produto encontrado</div>';
            }
            $('#resultado-produtos').html(html).show();
        }
    });
}

function adicionarProduto(id, nome, preco, estoque) {
    // Verificar se já existe no carrinho
    const index = carrinho.findIndex(p => p.id === id);
    
    if (index >= 0) {
        // Se já existe, aumenta quantidade
        if (carrinho[index].quantidade < estoque) {
            carrinho[index].quantidade++;
        } else {
            alert('Estoque insuficiente!');
            return;
        }
    } else {
        // Adiciona novo produto
        if (estoque > 0) {
            carrinho.push({
                id: id,
                nome: nome,
                preco: preco,
                quantidade: 1,
                estoque: estoque
            });
        } else {
            alert('Produto sem estoque!');
            return;
        }
    }

    $('#busca-produto').val('');
    $('#resultado-produtos').hide();
    atualizarCarrinho();
}

function removerProduto(index) {
    carrinho.splice(index, 1);
    atualizarCarrinho();
}

function atualizarQuantidade(index, quantidade) {
    if (quantidade <= 0) {
        removerProduto(index);
        return;
    }

    if (quantidade > carrinho[index].estoque) {
        alert('Estoque insuficiente!');
        quantidade = carrinho[index].estoque;
    }

    carrinho[index].quantidade = quantidade;
    atualizarCarrinho();
}

function atualizarCarrinho() {
    let html = '';
    
    if (carrinho.length === 0) {
        html = '<tr id="carrinho-vazio"><td colspan="5" class="text-center text-muted py-4"><i class="fas fa-shopping-cart fa-2x mb-2"></i><br>Carrinho vazio</td></tr>';
    } else {
        carrinho.forEach(function(item, index) {
            const subtotal = item.preco * item.quantidade;
            html += `
                <tr>
                    <td>${item.nome}</td>
                    <td class="text-center">
                        <input type="number" class="form-control form-control-sm text-center" 
                            value="${item.quantidade}" min="1" max="${item.estoque}" 
                            onchange="atualizarQuantidade(${index}, parseInt(this.value))" 
                            style="width: 70px;">
                    </td>
                    <td class="text-end">R$ ${parseFloat(item.preco).toFixed(2).replace('.', ',')}</td>
                    <td class="text-end">R$ ${parseFloat(subtotal).toFixed(2).replace('.', ',')}</td>
                    <td class="text-center">
                        <button type="button" class="btn btn-sm btn-danger" onclick="removerProduto(${index})">
                            <i class="fas fa-trash"></i>
                        </button>
                    </td>
                </tr>
            `;
        });
    }

    $('#carrinho-tbody').html(html);
    calcularTotal();
    verificarFinalizacao();
}

function calcularTotal() {
    let subtotal = 0;
    carrinho.forEach(function(item) {
        subtotal += item.preco * item.quantidade;
    });

    const desconto = parseFloat($('#desconto-input').val()) || 0;
    const total = subtotal - desconto;

    $('#subtotal').text('R$ ' + subtotal.toFixed(2).replace('.', ','));
    $('#total').text('R$ ' + total.toFixed(2).replace('.', ','));
    $('#total-hidden').val(total);
}

function verificarFinalizacao() {
    const podeFinalizar = carrinho.length > 0 && 
                          clienteSelecionado !== null && 
                          $('#forma_pagamento').val() !== '';
    
    $('#btn-finalizar').prop('disabled', !podeFinalizar);
}

// Preparar dados do formulário antes de enviar
$('#form-venda').on('submit', function(e) {
    if (carrinho.length === 0) {
        e.preventDefault();
        alert('Adicione pelo menos um produto ao carrinho!');
        return false;
    }

    if (!clienteSelecionado) {
        e.preventDefault();
        alert('Selecione um cliente!');
        return false;
    }

    if ($('#forma_pagamento').val() === '') {
        e.preventDefault();
        alert('Selecione uma forma de pagamento!');
        return false;
    }

    // Limpar itens anteriores
    $('input[name^="itens"]').remove();

    // Preparar itens para envio
    carrinho.forEach(function(item, index) {
        $('#form-venda').append(`<input type="hidden" name="itens[${index}][produto_id]" value="${item.id}">`);
        $('#form-venda').append(`<input type="hidden" name="itens[${index}][quantidade]" value="${item.quantidade}">`);
        $('#form-venda').append(`<input type="hidden" name="itens[${index}][preco]" value="${item.preco}">`);
    });

    return true;
});

// Verificar ao mudar forma de pagamento
$('#forma_pagamento').on('change', verificarFinalizacao);
</script>
@endpush
@endsection

